<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

class OrganismoController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $organismos = \App\Models\Organismo::all();
        return view('admin.organismos.index', compact('organismos'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $organismos = \App\Models\Organismo::all();
        return view('admin.organismos.create', compact('organismos'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'nombre' => 'required|string|max:255',
            'descripcion' => 'nullable|string|max:255',
            'direccion' => 'nullable|string|max:255',
            'telefono' => 'nullable|string|max:255',
            'email' => 'nullable|email|max:255',
            'parent_id' => 'nullable|exists:organismos,id'
        ]);
        
        // Validación adicional: verificar que no se cree una referencia circular
        if ($validated['parent_id']) {
            $parentOrganismo = \App\Models\Organismo::find($validated['parent_id']);
            if ($parentOrganismo) {
                // Verificar que no se exceda el máximo de 4 niveles
                $level = $this->getOrganismoLevel($parentOrganismo);
                if ($level >= 4) {
                    return redirect()->back()
                        ->withErrors(['parent_id' => 'No se puede crear un organismo en el nivel 5. El máximo permitido es 4 niveles.'])
                        ->withInput();
                }
            }
        }
        
        \App\Models\Organismo::create($validated);
        return redirect()->route('admin.organismos.index')->with('success', 'Organismo creado correctamente.');
    }
    
    /**
     * Obtener el nivel de un organismo en la jerarquía
     */
    private function getOrganismoLevel($organismo, $level = 1)
    {
        if ($organismo->parent_id) {
            $parent = \App\Models\Organismo::find($organismo->parent_id);
            return $this->getOrganismoLevel($parent, $level + 1);
        }
        return $level;
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $organismo = \App\Models\Organismo::findOrFail($id);
        $organismos = \App\Models\Organismo::where('id', '!=', $id)->get();
        return view('admin.organismos.edit', compact('organismo', 'organismos'));
    }

    /**
     * Show the form for editing the specified resource via AJAX.
     */
    public function editAjax(string $id)
    {
        $organismo = \App\Models\Organismo::findOrFail($id);
        $organismos = \App\Models\Organismo::where('id', '!=', $id)->get();
        
        $content = view('admin.organismos.partials.edit-form', compact('organismo', 'organismos'))->render();
        
        return response()->json([
            'success' => true,
            'title' => 'Editar Organismo',
            'content' => $content
        ]);
    }

    /**
     * Display the specified resource via AJAX.
     */
    public function showAjax(string $id)
    {
        $organismo = \App\Models\Organismo::with(['parent', 'children', 'solicitudes'])->findOrFail($id);
        
        $content = view('admin.organismos.partials.show-details', compact('organismo'))->render();
        
        return response()->json([
            'success' => true,
            'title' => 'Detalles del Organismo',
            'content' => $content
        ]);
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        $organismo = \App\Models\Organismo::findOrFail($id);
        
        $validated = $request->validate([
            'nombre' => 'required|string|max:255',
            'descripcion' => 'nullable|string|max:255',
            'direccion' => 'nullable|string|max:255',
            'telefono' => 'nullable|string|max:255',
            'email' => 'nullable|email|max:255',
            'parent_id' => 'nullable|exists:organismos,id'
        ]);
        
        // Validaciones adicionales para evitar referencias circulares
        if ($validated['parent_id']) {
            // No puede ser su propio padre
            if ($validated['parent_id'] == $id) {
                $errorMsg = 'Un organismo no puede ser su propio padre.';
                if ($request->ajax()) {
                    return response()->json([
                        'success' => false,
                        'message' => $errorMsg
                    ]);
                }
                return redirect()->back()->withErrors(['parent_id' => $errorMsg])->withInput();
            }
            
            // No puede tener como padre a uno de sus descendientes
            if ($this->isDescendant($id, $validated['parent_id'])) {
                $errorMsg = 'No se puede crear una referencia circular. El organismo seleccionado es un descendiente de este organismo.';
                if ($request->ajax()) {
                    return response()->json([
                        'success' => false,
                        'message' => $errorMsg
                    ]);
                }
                return redirect()->back()->withErrors(['parent_id' => $errorMsg])->withInput();
            }
            
            // Verificar que no se exceda el máximo de 4 niveles
            $parentOrganismo = \App\Models\Organismo::find($validated['parent_id']);
            if ($parentOrganismo) {
                $level = $this->getOrganismoLevel($parentOrganismo);
                if ($level >= 4) {
                    $errorMsg = 'No se puede mover el organismo al nivel 5. El máximo permitido es 4 niveles.';
                    if ($request->ajax()) {
                        return response()->json([
                            'success' => false,
                            'message' => $errorMsg
                        ]);
                    }
                    return redirect()->back()->withErrors(['parent_id' => $errorMsg])->withInput();
                }
            }
        }
        
        $organismo->update($validated);
        
        if ($request->ajax()) {
            return response()->json([
                'success' => true,
                'message' => 'Organismo actualizado correctamente.'
            ]);
        }
        
        return redirect()->route('admin.organismos.index')->with('success', 'Organismo actualizado correctamente.');
    }
    
    /**
     * Verificar si un organismo es descendiente de otro
     */
    private function isDescendant($ancestorId, $descendantId)
    {
        $current = \App\Models\Organismo::find($descendantId);
        
        while ($current && $current->parent_id) {
            if ($current->parent_id == $ancestorId) {
                return true;
            }
            $current = \App\Models\Organismo::find($current->parent_id);
        }
        
        return false;
    }

    /**
     * Get children of a specific organismo via AJAX.
     */
    public function childrenAjax(string $id)
    {
        $children = \App\Models\Organismo::where('parent_id', $id)->orderBy('nombre')->get();
        
        return response()->json([
            'success' => true,
            'children' => $children
        ]);
    }

    /**
     * Get all organismos via AJAX.
     */
    public function getAllAjax()
    {
        $organismos = \App\Models\Organismo::all();
        
        return response()->json([
            'success' => true,
            'organismos' => $organismos
        ]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        $organismo = \App\Models\Organismo::findOrFail($id);
        $organismo->delete();
        
        return redirect()->route('admin.organismos.index')->with('success', 'Organismo eliminado correctamente.');
    }
}
