<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Solicitud;
use App\Services\NotificationService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Symfony\Component\HttpFoundation\Response;

class SolicitudController extends Controller
{
    public function index(Request $request)
    {
        $query = \App\Models\Solicitud::with(['tipoSolicitud', 'organismoRequeridor', 'gestor', 'gestiones']);
        
        // Filtrar por estado si se especifica
        if ($request->filled('estado')) {
            $query->where('estado', $request->estado);
        }
        
        // Ordenamiento inteligente: cerradas al final, luego por fecha según filtro
        $order = $request->get('order', 'recientes');
        
        $query->orderByRaw("
            CASE 
                WHEN estado = 'cerrada' THEN 1 
                ELSE 0 
            END ASC
        ");
        
        // Para solicitudes no cerradas, aplicar el orden seleccionado
        if ($order === 'antiguas') {
            $query->orderBy('created_at', 'asc');
        } else {
            $query->orderBy('created_at', 'desc');
        }
        
        // Paginación de 20 elementos
        $solicitudes = $query->paginate(20);
        
        // Cargar evidencias para cada solicitud
        $solicitudesConEvidencias = $solicitudes->map(function ($solicitud) {
            $solicitud->evidencias = $this->getEvidencias($solicitud->id);
            return $solicitud;
        });
        
        return view('admin.solicitudes.index', compact('solicitudes'));
    }

    public function gestionar($id)
    {
        $solicitud = Solicitud::findOrFail($id);
        
        // Verificar que la solicitud no esté cerrada
        if ($solicitud->estado === 'cerrada') {
            return redirect()->back()->with('error', 'No se puede gestionar una solicitud cerrada.');
        }
        
        // Verificar que no tenga gestor asignado
        if ($solicitud->gestor_id) {
            $gestorActual = $solicitud->gestor->name ?? 'Usuario desconocido';
            return redirect()->back()->with('error', "Esta solicitud ya está siendo gestionada por {$gestorActual}.");
        }
        
        // Asignar al usuario actual como gestor
        $solicitud->gestor_id = Auth::id();
        $solicitud->fecha_asignacion = now();
        $solicitud->estado = 'en_gestion';
        $solicitud->save();
        
        return redirect()->route('admin.solicitudes.show', $solicitud->id)->with('success', 'Ahora está gestionando esta solicitud.');
    }

    public function show($id)
    {
        $solicitud = Solicitud::with(['tipoSolicitud', 'organismoRequeridor', 'gestor', 'gestiones'])->findOrFail($id);
        $evidencias = $this->getEvidencias($id);
        return view('admin.solicitudes.show', compact('solicitud', 'evidencias'));
    }

    public function cerrar(Request $request, $id, NotificationService $notificationService)
    {
        $request->validate([
            'observaciones_cierre' => 'required|string|min:10|max:1000',
            'archivos_cierre.*' => 'nullable|file|extensions:pdf,doc,docx,jpg,jpeg,png|max:10240', // 10MB max
        ], [
            'observaciones_cierre.required' => 'Las observaciones de cierre son obligatorias.',
            'observaciones_cierre.min' => 'Las observaciones deben tener al menos 10 caracteres.',
            'observaciones_cierre.max' => 'Las observaciones no pueden exceder 1000 caracteres.',
            'archivos_cierre.*.file' => 'Uno de los archivos seleccionados no es válido.',
            'archivos_cierre.*.extensions' => 'Los archivos deben tener extensión: PDF, DOC, DOCX, JPG, JPEG o PNG.',
            'archivos_cierre.*.max' => 'Cada archivo no puede exceder 10MB de tamaño.',
        ]);

        $solicitud = Solicitud::findOrFail($id);
        
        // Verificar que la solicitud puede ser cerrada
        if ($solicitud->estado === 'cerrada') {
            return redirect()->back()->with('error', 'Esta solicitud ya está cerrada.');
        }
        
        // VALIDACIÓN DE SEGURIDAD: Solo el gestor asignado puede cerrar la solicitud
        if ($solicitud->gestor_id !== auth()->id()) {
            return redirect()->back()->with('error', 'Solo el gestor asignado puede cerrar esta solicitud.');
        }

        // Procesar archivos de cierre si existen
        $uriArchivosCierre = null;
        if ($request->hasFile('archivos_cierre')) {
            $uriArchivosCierre = $this->procesarArchivosCierre($request->file('archivos_cierre'), $solicitud->id);
        }

        // Actualizar la solicitud
        $solicitud->estado = 'cerrada';
        $solicitud->fecha_cierre = now();
        $solicitud->observaciones_cierre = $request->observaciones_cierre;
        $solicitud->uri_archivos_cierre = $uriArchivosCierre;
        
        $solicitud->save();

        // Enviar notificación por correo al solicitante
        try {
            $notificationService->notificarCierreSolicitud($solicitud, Auth::user());
            $mensaje = 'Solicitud cerrada correctamente. Se ha enviado una notificación al solicitante.';
        } catch (\Exception $e) {
            Log::error('Error al enviar notificación de cierre: ' . $e->getMessage());
            $mensaje = 'Solicitud cerrada correctamente, pero hubo un problema al enviar la notificación por correo.';
        }

        return redirect()->route('admin.solicitudes.index')->with('success', $mensaje);
    }

    public function storeGestion(Request $request, $id)
    {
        $solicitud = \App\Models\Solicitud::findOrFail($id);
        $request->validate([
            'descripcion' => 'required|string|max:1000',
            'archivos_gestion.*' => 'nullable|file|extensions:pdf,doc,docx,jpg,jpeg,png,gif|max:10240', // 10MB max
        ], [
            'descripcion.required' => 'La descripción de la gestión es obligatoria.',
            'descripcion.max' => 'La descripción no puede exceder 1000 caracteres.',
            'archivos_gestion.*.file' => 'Uno de los archivos seleccionados no es válido.',
            'archivos_gestion.*.extensions' => 'Los archivos deben tener extensión: PDF, DOC, DOCX, JPG, JPEG, PNG o GIF.',
            'archivos_gestion.*.max' => 'Cada archivo no puede exceder 10MB de tamaño.',
        ]);

        // Procesar archivos adjuntos si existen
        $uriArchivosAdjuntos = null;
        if ($request->hasFile('archivos_gestion')) {
            $uriArchivosAdjuntos = $this->procesarArchivosGestion($request->file('archivos_gestion'), $solicitud->id);
        }

        $solicitud->gestiones()->create([
            'usuario_id' => auth()->id(),
            'descripcion_gestion' => $request->descripcion,
            'fecha_gestion' => now(),
            'accion_realizada' => $request->accion_realizada ?? $request->descripcion,
            'uri_archivos_adjuntos' => $uriArchivosAdjuntos,
        ]);
        
        // Si viene de un modal (detectamos por el referer o un campo oculto)
        if ($request->has('from_modal') || str_contains($request->header('referer', ''), 'admin/solicitudes')) {
            return redirect()->route('admin.solicitudes.index')->with('success', 'Acción registrada correctamente.');
        }
        
        return redirect()->route('admin.solicitudes.show', $solicitud->id)->with('success', 'Acción registrada correctamente.');
    }

    /**
     * Obtener la lista de archivos de evidencia para una solicitud
     */
    public function getEvidencias($id)
    {
        $solicitud = Solicitud::findOrFail($id);

        if (!$solicitud->uri_evidencia || !file_exists($solicitud->uri_evidencia)) {
            return [];
        }

        $files = [];
        $evidenciaPath = $solicitud->uri_evidencia;

        if (is_dir($evidenciaPath)) {
            $scanFiles = scandir($evidenciaPath);
            foreach ($scanFiles as $file) {
                if ($file !== '.' && $file !== '..' && is_file($evidenciaPath . '/' . $file)) {
                    $filePath = $evidenciaPath . '/' . $file;
                    $files[] = [
                        'name' => $file,
                        'path' => $filePath,
                        'size' => filesize($filePath),
                        'modified' => date('d/m/Y H:i', filemtime($filePath)),
                        'download_url' => route('admin.solicitudes.download-evidencia', ['id' => $id, 'filename' => $file])
                    ];
                }
            }
        }

        return $files;
    }

    /**
     * Descargar un archivo de evidencia específico
     */
    public function downloadEvidencia($id, $filename)
    {
        $solicitud = Solicitud::findOrFail($id);

        if (!$solicitud->uri_evidencia || !file_exists($solicitud->uri_evidencia)) {
            abort(404, 'Carpeta de evidencias no encontrada');
        }

        $filePath = $solicitud->uri_evidencia . '/' . $filename;

        // Validar que el archivo existe y está dentro de la carpeta de evidencias
        if (!file_exists($filePath) || !str_starts_with(realpath($filePath), realpath($solicitud->uri_evidencia))) {
            abort(404, 'Archivo no encontrado');
        }

        // Determinar el tipo MIME del archivo
        $mimeType = mime_content_type($filePath);

        return response()->download($filePath, $filename, [
            'Content-Type' => $mimeType,
        ]);
    }

    /**
     * Descargar todas las evidencias en un archivo ZIP
     */
    public function downloadAllEvidencias($id)
    {
        $solicitud = Solicitud::findOrFail($id);

        if (!$solicitud->uri_evidencia || !file_exists($solicitud->uri_evidencia)) {
            abort(404, 'Carpeta de evidencias no encontrada');
        }

        $evidenciaPath = $solicitud->uri_evidencia;
        $zipFileName = "evidencias_solicitud_{$id}.zip";
        $zipPath = storage_path('app/temp/' . $zipFileName);

        // Crear directorio temporal si no existe
        if (!file_exists(dirname($zipPath))) {
            mkdir(dirname($zipPath), 0755, true);
        }

        $zip = new \ZipArchive();

        if ($zip->open($zipPath, \ZipArchive::CREATE | \ZipArchive::OVERWRITE) === TRUE) {
            $files = scandir($evidenciaPath);
            foreach ($files as $file) {
                if ($file !== '.' && $file !== '..' && is_file($evidenciaPath . '/' . $file)) {
                    $zip->addFile($evidenciaPath . '/' . $file, $file);
                }
            }
            $zip->close();

            return response()->download($zipPath)->deleteFileAfterSend(true);
        } else {
            abort(500, 'No se pudo crear el archivo ZIP');
        }
    }

    /**
     * Procesar archivos de cierre de solicitud
     */
    private function procesarArchivosCierre($archivos, $solicitudId)
    {
        $basePath = env('ARCHIVOS_CIERRE_PATH', storage_path('app/archivos_cierre'));
        $solicitudPath = $basePath . '/' . $solicitudId;

        // Crear directorio si no existe
        if (!file_exists($solicitudPath)) {
            mkdir($solicitudPath, 0755, true);
        }

        foreach ($archivos as $archivo) {
            $nombreOriginal = $archivo->getClientOriginalName();
            $extension = $archivo->getClientOriginalExtension();
            $nombreSeguro = time() . '_' . preg_replace('/[^A-Za-z0-9\-\.]/', '_', $nombreOriginal);
            
            $archivo->move($solicitudPath, $nombreSeguro);
        }

        return $solicitudPath;
    }

    /**
     * Procesar archivos adjuntos de gestion
     */
    private function procesarArchivosGestion($archivos, $solicitudId)
    {
        // Guardar en la raíz de la aplicación
        $basePath = base_path('evidencias_gestion');
        $solicitudPath = $basePath . '/' . $solicitudId;
        $gestionPath = $solicitudPath . '/' . time(); // Carpeta única por gestión

        // Crear directorio si no existe
        if (!file_exists($gestionPath)) {
            mkdir($gestionPath, 0755, true);
        }

        foreach ($archivos as $archivo) {
            $nombreOriginal = $archivo->getClientOriginalName();
            $extension = $archivo->getClientOriginalExtension();
            $nombreSeguro = time() . '_' . preg_replace('/[^A-Za-z0-9\-\.]/', '_', $nombreOriginal);
            $archivo->move($gestionPath, $nombreSeguro);
        }

        // Guardar la ruta relativa para la base de datos
        $relativePath = 'evidencias_gestion/' . $solicitudId . '/' . basename($gestionPath);
        return $relativePath;
    }

    /**
     * Descargar archivo adjunto de gestión
     */
    public function downloadGestionArchivo($gestionId, $filename)
    {
        $gestion = \App\Models\Gestion::findOrFail($gestionId);
        $basePath = base_path($gestion->uri_archivos_adjuntos);
        $filePath = $basePath . '/' . $filename;

        // Validar que el archivo existe y está dentro de la carpeta de gestión
        if (!file_exists($filePath) || !str_starts_with(realpath($filePath), realpath($basePath))) {
            abort(404, 'Archivo no encontrado');
        }

        $mimeType = mime_content_type($filePath);
        return response()->download($filePath, $filename, [
            'Content-Type' => $mimeType,
        ]);
    }
}
