<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;

class Solicitud extends Model
{
    protected $fillable = [
        'codigo_seguimiento',
        'tipo_solicitud_id',
        'organismo_requeridor_id',
        'notificacion',
        'detalle_solicitud',
        'fecha_solicitud',
        'observaciones',
        'estado',
        'uri_evidencia',
        // Datos de contacto
        'nombres',
        'apellido_paterno',
        'apellido_materno',
        'nacionalidad',
        'pais_residencia',
        'calle',
        'numero',
        'villa',
        'comuna',
        'telefono',
        'email',
        // Datos estadísticos
        'tipo_zona',
        'educacion',
        'edad',
        'ocupacion',
        'genero',
        'organizacion',
        'apoderado',
        'usuario_respuesta',
        // Gestión
        'gestor_id',
        'fecha_asignacion',
        'fecha_cierre',
        'tiempo_limite_horas',
        'observaciones_cierre',
        'uri_archivos_cierre',
    ];
    /**
     * Usuario gestor (moderador que gestiona la solicitud)
     */
    public function gestor()
    {
        return $this->belongsTo(User::class, 'gestor_id');
    }

    protected $table = 'solicitudes';
    /**
     * The attributes that should be hidden for serialization.
     *
     * @var list<string>
     */
    protected $hidden = [
        'created_at',
        'updated_at',
    ];

    protected $casts = [
        'notificacion' => 'boolean',
        'fecha_solicitud' => 'date',
    ];

    public function gestiones()
    {
        return $this->hasMany(Gestion::class, 'solicitud_id');
    }

    public function tipoSolicitud()
    {
        return $this->belongsTo(TipoSolicitud::class, 'tipo_solicitud_id');
    }
    public function organismoRequeridor()
    {
        return $this->belongsTo(Organismo::class, 'organismo_requeridor_id');
    }

    /**
     * Generar código de seguimiento único
     */
    public static function generarCodigoSeguimiento(): string
    {
        do {
            // Generar código de 12 caracteres: SIAC-YYYYMMDD-XXXX
            $fecha = date('Ymd');
            $random = strtoupper(substr(str_shuffle('ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789'), 0, 4));
            $codigo = "SIAC-{$fecha}-{$random}";
        } while (self::where('codigo_seguimiento', $codigo)->exists());
        
        return $codigo;
    }

    /**
     * Obtener archivos de cierre
     */
    public function getArchivosCierre(): array
    {
        if (!$this->uri_archivos_cierre || !file_exists($this->uri_archivos_cierre)) {
            return [];
        }

        $files = [];
        if (is_dir($this->uri_archivos_cierre)) {
            $scanFiles = scandir($this->uri_archivos_cierre);
            foreach ($scanFiles as $file) {
                if ($file !== '.' && $file !== '..' && is_file($this->uri_archivos_cierre . '/' . $file)) {
                    $filePath = $this->uri_archivos_cierre . '/' . $file;
                    $files[] = [
                        'name' => $file,
                        'path' => $filePath,
                        'size' => filesize($filePath),
                        'modified' => date('d/m/Y H:i', filemtime($filePath)),
                    ];
                }
            }
        }
        return $files;
    }
}
