<?php

namespace App\Services;

use App\Models\Solicitud;
use App\Models\User;
use App\Mail\SolicitudCerradaMail;
use App\Mail\SolicitudCreadaMail;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Log;

class NotificationService
{
    /**
     * Envía notificación cuando una solicitud es cerrada
     */
    public function notificarCierreSolicitud(Solicitud $solicitud, User $gestorQueCierra = null)
    {
        try {
            // Verificar que la solicitud tenga un email de solicitante
            if (!$solicitud->email) {
                Log::warning("Solicitud #{$solicitud->id} no tiene email de solicitante para notificar");
                return false;
            }

            // Construir nombre completo del solicitante
            $nombreCompleto = trim($solicitud->nombres . ' ' . $solicitud->apellido_paterno . ' ' . $solicitud->apellido_materno);
            if (empty($nombreCompleto)) {
                $nombreCompleto = 'Usuario';
            }

            // Datos para el template
            $datosNotificacion = [
                'solicitud' => $solicitud,
                'solicitante_nombre' => $nombreCompleto,
                'gestor' => $gestorQueCierra ?? $solicitud->gestor,
                'fecha_cierre' => $solicitud->fecha_cierre ?? now(),
                'tiempo_resolucion' => $this->calcularTiempoResolucion($solicitud),
                'tipo_solicitud' => $solicitud->tipoSolicitud->denominacion ?? 'N/A',
                'organismo' => $solicitud->organismoRequeridor->nombre ?? 'N/A',
                'codigo_seguimiento' => $solicitud->codigo_seguimiento,
                'url_seguimiento' => route('seguimiento.mostrar', $solicitud->codigo_seguimiento),
            ];

            // Enviar el correo
            Mail::to($solicitud->email)
                ->send(new SolicitudCerradaMail($datosNotificacion));

            Log::info("Notificación de cierre enviada para solicitud #{$solicitud->id} a {$solicitud->email}");
            
            return true;

        } catch (\Exception $e) {
            Log::error("Error al enviar notificación de cierre para solicitud #{$solicitud->id}: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Envía notificación cuando una solicitud es asignada a un gestor
     */
    public function notificarAsignacionSolicitud(Solicitud $solicitud, User $gestor)
    {
        try {
            // TODO: Implementar template de asignación
            Log::info("Notificación de asignación pendiente para solicitud #{$solicitud->id}");
            return true;
        } catch (\Exception $e) {
            Log::error("Error al enviar notificación de asignación: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Envía notificación cuando una solicitud está próxima a vencer
     */
    public function notificarSolicitudProximaVencer(Solicitud $solicitud)
    {
        try {
            // TODO: Implementar template de vencimiento próximo
            Log::info("Notificación de vencimiento próximo pendiente para solicitud #{$solicitud->id}");
            return true;
        } catch (\Exception $e) {
            Log::error("Error al enviar notificación de vencimiento: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Envía notificación cuando se crea una nueva solicitud
     */
    public function notificarCreacionSolicitud(Solicitud $solicitud)
    {
        try {
            // Verificar que la solicitud tenga email y que el usuario quiera notificaciones
            if (!$solicitud->email || !$solicitud->notificacion) {
                Log::info("Solicitud #{$solicitud->id} no tiene email o no quiere notificaciones");
                return false;
            }

            // Construir nombre completo del solicitante
            $nombreCompleto = trim($solicitud->nombres . ' ' . $solicitud->apellido_paterno . ' ' . $solicitud->apellido_materno);
            if (empty($nombreCompleto)) {
                $nombreCompleto = 'Usuario';
            }

            // Calcular tiempo límite en días
            $tiempoLimiteDias = null;
            if ($solicitud->tiempo_limite_horas) {
                $tiempoLimiteDias = ceil($solicitud->tiempo_limite_horas / 24);
            }

            // Datos para el template
            $datosNotificacion = [
                'solicitud' => $solicitud,
                'solicitante_nombre' => $nombreCompleto,
                'tipo_solicitud' => $solicitud->tipoSolicitud->denominacion ?? 'N/A',
                'organismo' => $solicitud->organismoRequeridor->nombre ?? 'N/A',
                'tiempo_limite_dias' => $tiempoLimiteDias,
                'codigo_seguimiento' => $solicitud->codigo_seguimiento,
                'url_seguimiento' => route('seguimiento.mostrar', $solicitud->codigo_seguimiento),
            ];

            // Enviar el correo
            Mail::to($solicitud->email)
                ->send(new SolicitudCreadaMail($datosNotificacion));

            Log::info("Notificación de creación enviada para solicitud #{$solicitud->id} a {$solicitud->email}");
            
            return true;

        } catch (\Exception $e) {
            Log::error("Error al enviar notificación de creación para solicitud #{$solicitud->id}: " . $e->getMessage());
            return false;
        }
    }

    /**
     * Calcula el tiempo de resolución de una solicitud
     */
    private function calcularTiempoResolucion(Solicitud $solicitud)
    {
        if (!$solicitud->fecha_asignacion || !$solicitud->fecha_cierre) {
            return 'N/A';
        }

        $inicio = \Carbon\Carbon::parse($solicitud->fecha_asignacion);
        $fin = \Carbon\Carbon::parse($solicitud->fecha_cierre);
        $horas = $inicio->diffInHours($fin);
        
        $dias = ceil($horas / 24);
        
        if ($dias == 0) {
            $dias = 1;
        }
        
        return $dias == 1 ? "1 día" : "{$dias} días";
    }

    /**
     * Obtiene destinatarios adicionales según el tipo de notificación
     */
    private function obtenerDestinatariosAdicionales($tipoNotificacion, Solicitud $solicitud)
    {
        $destinatarios = [];

        switch ($tipoNotificacion) {
            case 'cierre_urgente':
                // Notificar también a supervisores si es urgente
                $destinatarios = User::where('role', 'supervisor')->pluck('email')->toArray();
                break;
            
            case 'vencimiento':
                // Notificar al gestor y su supervisor
                if ($solicitud->gestor) {
                    $destinatarios[] = $solicitud->gestor->email;
                }
                break;
        }

        return $destinatarios;
    }
}
