/**
 * =========================================
 * ADMIN 404 - JavaScript para animaciones
 * =========================================
 */

document.addEventListener('DOMContentLoaded', function() {
    initializeAdmin404();
});

/**
 * Inicializar todas las funcionalidades de la página 404
 */
function initializeAdmin404() {
    createDynamicParticles();
    addMouseInteraction();
    addKeyboardNavigation();
    startNumberAnimation();
    addRandomGlitchEffect();
}

/**
 * Crear partículas dinámicas
 */
function createDynamicParticles() {
    const container = document.querySelector('.admin-404-container');
    const particlesContainer = document.querySelector('.admin-404-particles');
    
    if (!particlesContainer) return;
    
    // Crear partículas adicionales dinámicas
    for (let i = 0; i < 15; i++) {
        createRandomParticle(particlesContainer, i);
    }
    
    // Crear nueva partícula cada 3 segundos
    setInterval(() => {
        createRandomParticle(particlesContainer, Math.random());
    }, 3000);
}

/**
 * Crear una partícula aleatoria
 */
function createRandomParticle(container, index) {
    const particle = document.createElement('div');
    particle.className = 'particle dynamic-particle';
    
    // Propiedades aleatorias
    const size = Math.random() * 8 + 3; // 3-11px
    const left = Math.random() * 100; // 0-100%
    const duration = Math.random() * 10 + 8; // 8-18s
    const delay = Math.random() * 5; // 0-5s
    
    particle.style.cssText = `
        left: ${left}%;
        width: ${size}px;
        height: ${size}px;
        animation: floatUp ${duration}s infinite linear;
        animation-delay: ${delay}s;
        background: rgba(255, 255, 255, ${Math.random() * 0.3 + 0.1});
    `;
    
    container.appendChild(particle);
    
    // Remover partícula después de la animación
    setTimeout(() => {
        if (particle.parentNode) {
            particle.parentNode.removeChild(particle);
        }
    }, (duration + delay) * 1000);
}

/**
 * Agregar interacción con el mouse
 */
function addMouseInteraction() {
    const number = document.querySelector('.admin-404-number');
    const container = document.querySelector('.admin-404-container');
    
    if (!number || !container) return;
    
    container.addEventListener('mousemove', function(e) {
        const rect = container.getBoundingClientRect();
        const x = (e.clientX - rect.left) / rect.width;
        const y = (e.clientY - rect.top) / rect.height;
        
        // Efecto parallax sutil en el número
        const moveX = (x - 0.5) * 20;
        const moveY = (y - 0.5) * 20;
        
        number.style.transform = `translate(${moveX}px, ${moveY}px)`;
        
        // Crear partícula en la posición del mouse
        if (Math.random() < 0.1) { // 10% de probabilidad
            createMouseParticle(e.clientX, e.clientY);
        }
    });
    
    container.addEventListener('mouseleave', function() {
        number.style.transform = 'translate(0, 0)';
    });
}

/**
 * Crear partícula en la posición del mouse
 */
function createMouseParticle(x, y) {
    const particle = document.createElement('div');
    particle.style.cssText = `
        position: fixed;
        left: ${x}px;
        top: ${y}px;
        width: 4px;
        height: 4px;
        background: rgba(255, 255, 255, 0.8);
        border-radius: 50%;
        pointer-events: none;
        z-index: 1000;
        animation: mouseParticleFloat 2s ease-out forwards;
    `;
    
    document.body.appendChild(particle);
    
    setTimeout(() => {
        if (particle.parentNode) {
            particle.parentNode.removeChild(particle);
        }
    }, 2000);
}

// CSS para animación de partícula del mouse
const mouseParticleStyle = document.createElement('style');
mouseParticleStyle.textContent = `
    @keyframes mouseParticleFloat {
        0% {
            opacity: 1;
            transform: translate(-50%, -50%) scale(1);
        }
        100% {
            opacity: 0;
            transform: translate(-50%, -150px) scale(0);
        }
    }
`;
document.head.appendChild(mouseParticleStyle);

/**
 * Agregar navegación por teclado
 */
function addKeyboardNavigation() {
    document.addEventListener('keydown', function(e) {
        switch(e.key) {
            case 'h':
            case 'H':
                // Ir al dashboard
                const dashboardBtn = document.querySelector('[href*="admin"]');
                if (dashboardBtn) {
                    dashboardBtn.click();
                }
                break;
            case 'b':
            case 'B':
                // Volver atrás
                window.history.back();
                break;
            case 'r':
            case 'R':
                // Recargar página
                window.location.reload();
                break;
            case 'Escape':
                // Ir al dashboard
                const escapeBtn = document.querySelector('[href*="admin"]');
                if (escapeBtn) {
                    escapeBtn.click();
                }
                break;
        }
    });
}

/**
 * Animación del número 404
 */
function startNumberAnimation() {
    const number = document.querySelector('.admin-404-number');
    if (!number) return;
    
    // Efecto de conteo animado
    let currentNumber = 0;
    const targetNumber = 404;
    const increment = 13;
    const duration = 1500; // 1.5 segundos
    const intervalTime = duration / (targetNumber / increment);
    
    const countInterval = setInterval(() => {
        currentNumber += increment;
        if (currentNumber >= targetNumber) {
            currentNumber = targetNumber;
            clearInterval(countInterval);
            
            // Agregar efecto de "temblor" al final
            number.style.animation += ', shake 0.5s ease-in-out';
        }
        number.textContent = currentNumber;
    }, intervalTime);
}

// CSS para efecto de temblor
const shakeStyle = document.createElement('style');
shakeStyle.textContent = `
    @keyframes shake {
        0%, 100% { transform: translateX(0); }
        10%, 30%, 50%, 70%, 90% { transform: translateX(-5px); }
        20%, 40%, 60%, 80% { transform: translateX(5px); }
    }
`;
document.head.appendChild(shakeStyle);

/**
 * Efecto de glitch aleatorio
 */
function addRandomGlitchEffect() {
    const title = document.querySelector('.admin-404-title');
    const subtitle = document.querySelector('.admin-404-subtitle');
    
    if (!title || !subtitle) return;
    
    // Aplicar glitch cada 8-15 segundos
    function scheduleGlitch() {
        const delay = Math.random() * 7000 + 8000; // 8-15 segundos
        
        setTimeout(() => {
            applyGlitchEffect(Math.random() < 0.5 ? title : subtitle);
            scheduleGlitch(); // Programar siguiente glitch
        }, delay);
    }
    
    scheduleGlitch();
}

/**
 * Aplicar efecto glitch a un elemento
 */
function applyGlitchEffect(element) {
    const originalText = element.textContent;
    const glitchChars = '!@#$%^&*()_+-=[]{}|;:,.<>?';
    let glitchCount = 0;
    const maxGlitches = 5;
    
    const glitchInterval = setInterval(() => {
        // Crear texto con glitch
        let glitchedText = '';
        for (let i = 0; i < originalText.length; i++) {
            if (Math.random() < 0.3 && originalText[i] !== ' ') {
                glitchedText += glitchChars[Math.floor(Math.random() * glitchChars.length)];
            } else {
                glitchedText += originalText[i];
            }
        }
        
        element.textContent = glitchedText;
        glitchCount++;
        
        if (glitchCount >= maxGlitches) {
            clearInterval(glitchInterval);
            element.textContent = originalText; // Restaurar texto original
        }
    }, 100);
}

/**
 * Efecto de pulsación en botones
 */
function addButtonPulseEffect() {
    const buttons = document.querySelectorAll('.admin-404-btn');
    
    buttons.forEach(button => {
        button.addEventListener('mouseenter', function() {
            this.style.animation += ', pulse 0.6s ease-in-out';
        });
        
        button.addEventListener('animationend', function() {
            // Limpiar animaciones después de completarse
            this.style.animation = this.style.animation.replace(', pulse 0.6s ease-in-out', '');
        });
    });
}

// CSS para efecto de pulsación
const pulseStyle = document.createElement('style');
pulseStyle.textContent = `
    @keyframes pulse {
        0% { transform: scale(1); }
        50% { transform: scale(1.05); }
        100% { transform: scale(1); }
    }
`;
document.head.appendChild(pulseStyle);

// Inicializar efecto de pulsación cuando se carga la página
document.addEventListener('DOMContentLoaded', addButtonPulseEffect);

/**
 * Mostrar información de debug (solo en desarrollo)
 */
function showDebugInfo() {
    if (window.location.hostname === 'localhost' || window.location.hostname === '127.0.0.1') {
        console.log('🎮 Admin 404 Page - Controles disponibles:');
        console.log('• Presiona "H" para ir al dashboard');
        console.log('• Presiona "B" para volver atrás');
        console.log('• Presiona "R" para recargar');
        console.log('• Presiona "Escape" para ir al dashboard');
        console.log('• Mueve el mouse para efectos visuales');
    }
}

// Mostrar debug info al cargar
document.addEventListener('DOMContentLoaded', showDebugInfo);
