<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class CleanTempEvidencias extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'evidencias:clean-temp {--hours=24 : Horas después de las cuales eliminar archivos temporales}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Limpia archivos temporales de evidencias que no fueron procesados';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $hours = $this->option('hours');
        $tempDir = storage_path('app/temp_evidencias');
        
        if (!is_dir($tempDir)) {
            $this->info('El directorio de archivos temporales no existe.');
            return 0;
        }
        
        $cutoffTime = time() - ($hours * 3600);
        $deletedCount = 0;
        $totalSize = 0;
        
        $files = scandir($tempDir);
        foreach ($files as $file) {
            if ($file === '.' || $file === '..') {
                continue;
            }
            
            $filePath = $tempDir . '/' . $file;
            if (is_file($filePath) && filemtime($filePath) < $cutoffTime) {
                $size = filesize($filePath);
                $totalSize += $size;
                
                if (unlink($filePath)) {
                    $deletedCount++;
                    Log::info("Archivo temporal eliminado: {$file}");
                } else {
                    Log::error("No se pudo eliminar archivo temporal: {$file}");
                }
            }
        }
        
        $totalSizeMB = round($totalSize / 1024 / 1024, 2);
        
        $this->info("Limpieza completada:");
        $this->info("- Archivos eliminados: {$deletedCount}");
        $this->info("- Espacio liberado: {$totalSizeMB} MB");
        
        Log::info("Limpieza de archivos temporales completada", [
            'deleted_count' => $deletedCount,
            'total_size_mb' => $totalSizeMB,
            'hours_threshold' => $hours
        ]);
        
        return 0;
    }
}
