<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Mail;
use App\Mail\SolicitudCerradaMail;
use App\Mail\SolicitudCreadaMail;

class TestEmailCommand extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'siac:test-email {email} {--type=simple : Tipo de correo (simple, solicitud, creada)}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Prueba el envío de correos del sistema SIAC';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $email = $this->argument('email');
        $type = $this->option('type');

        $this->info("🚀 Iniciando prueba de correo...");
        $this->info("📧 Destinatario: {$email}");
        $this->info("📝 Tipo: {$type}");
        
        try {
            if ($type === 'simple') {
                $this->sendSimpleEmail($email);
            } elseif ($type === 'solicitud') {
                $this->sendSolicitudEmail($email);
            } elseif ($type === 'creada') {
                $this->sendSolicitudCreadaEmail($email);
            } else {
                $this->error("❌ Tipo de correo no válido. Usa: simple, solicitud o creada");
                return 1;
            }

            $this->info("✅ Correo enviado exitosamente!");
            
            if (config('mail.default') === 'log') {
                $this->warn("⚠️  Configuración actual: LOG");
                $this->info("📁 Revisa el archivo: storage/logs/laravel.log");
                $this->info("🔍 Comando útil: tail -f storage/logs/laravel.log");
            }
            
            return 0;
            
        } catch (\Exception $e) {
            $this->error("❌ Error al enviar correo:");
            $this->error($e->getMessage());
            
            $this->warn("\n💡 Posibles soluciones:");
            $this->info("1. Verifica la configuración en .env");
            $this->info("2. Asegúrate de que las credenciales sean correctas");
            $this->info("3. Revisa que el servidor SMTP esté accesible");
            $this->info("4. Para Gmail, usa una contraseña de aplicación");
            
            return 1;
        }
    }

    private function sendSimpleEmail($email)
    {
        $this->info("📤 Enviando correo simple...");
        
        Mail::raw('¡Hola! Este es un correo de prueba del Sistema SIAC. Si recibes este mensaje, la configuración de correo está funcionando correctamente.', function ($message) use ($email) {
            $message->to($email)
                    ->subject('✅ Prueba de Correo - Sistema SIAC')
                    ->from(config('mail.from.address'), config('mail.from.name'));
        });
    }

    private function sendSolicitudEmail($email)
    {
        $this->info("📤 Enviando correo de solicitud cerrada...");
        
        // Crear datos de prueba
        $solicitudPrueba = new \stdClass();
        $solicitudPrueba->id = 999;
        $solicitudPrueba->email = $email;
        $solicitudPrueba->created_at = now();
        $solicitudPrueba->observaciones_cierre = 'Esta es una solicitud de prueba generada automáticamente para verificar el funcionamiento del sistema de notificaciones.';

        $gestorPrueba = new \stdClass();
        $gestorPrueba->name = 'Sistema de Pruebas';

        $datos = [
            'solicitud' => $solicitudPrueba,
            'solicitante_nombre' => 'Usuario de Prueba',
            'gestor' => $gestorPrueba,
            'fecha_cierre' => now(),
            'tiempo_resolucion' => '2.5 días',
            'tipo_solicitud' => 'Solicitud de Prueba',
            'organismo' => 'Organismo de Pruebas SIAC',
        ];

        Mail::to($email)->send(new SolicitudCerradaMail($datos));
    }

    private function sendSolicitudCreadaEmail($email)
    {
        $this->info("📤 Enviando correo de solicitud creada...");
        
        // Crear datos de prueba
        $solicitudPrueba = new \stdClass();
        $solicitudPrueba->id = 888;
        $solicitudPrueba->email = $email;
        $solicitudPrueba->created_at = now();
        $solicitudPrueba->detalle_solicitud = 'Esta es una solicitud de prueba para verificar el envío de notificaciones de creación.';
        $solicitudPrueba->notificacion = true;

        $datos = [
            'solicitud' => $solicitudPrueba,
            'solicitante_nombre' => 'Usuario de Prueba',
            'tipo_solicitud' => 'Solicitud de Prueba - Creación',
            'organismo' => 'Organismo de Pruebas SIAC',
            'tiempo_limite_dias' => 7,
        ];

        Mail::to($email)->send(new SolicitudCreadaMail($datos));
    }
}
