<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Solicitud;
use App\Models\User;
use App\Models\Gestion;
use App\Models\TipoSolicitud;
use App\Models\Organismo;
use Illuminate\Http\Request;
use Carbon\Carbon;
use Illuminate\Support\Facades\DB;

class DashboardController extends Controller
{
    public function index()
    {
        // Estadísticas generales
        $estadisticas = $this->getEstadisticasGenerales();
        
        // Solicitudes urgentes (menos de 2 días restantes)
        $solicitudesUrgentes = $this->getSolicitudesUrgentes();
        
        // Top gestores con más solicitudes activas
        $topGestores = $this->getTopGestores();
        
        // Últimas solicitudes creadas
        $ultimasSolicitudes = $this->getUltimasSolicitudes();
        
        // Estadísticas por estado
        $estadisticasPorEstado = $this->getEstadisticasPorEstado();
        
        // Tipos de solicitud más comunes
        $tiposSolicitudComunes = $this->getTiposSolicitudComunes();
        
        // Organismos con más solicitudes
        $organismosActivos = $this->getOrganismosActivos();
        
        // Tendencia de solicitudes (últimos 7 días)
        $tendenciaSolicitudes = $this->getTendenciaSolicitudes();
        
        // Tiempo promedio de resolución
        $tiempoPromedioResolucion = $this->getTiempoPromedioResolucion();
        
        return view('admin.dashboard', compact(
            'estadisticas',
            'solicitudesUrgentes',
            'topGestores',
            'ultimasSolicitudes',
            'estadisticasPorEstado',
            'tiposSolicitudComunes',
            'organismosActivos',
            'tendenciaSolicitudes',
            'tiempoPromedioResolucion'
        ));
    }
    
    private function getEstadisticasGenerales()
    {
        $total = Solicitud::count();
        $pendientes = Solicitud::where('estado', 'pendiente')->count();
        $enGestion = Solicitud::where('estado', 'en_gestion')->count();
        $cerradas = Solicitud::where('estado', 'cerrada')->count();
        $hoy = Solicitud::whereDate('created_at', today())->count();
        $estaSemana = Solicitud::whereBetween('created_at', [
            Carbon::now()->startOfWeek(),
            Carbon::now()->endOfWeek()
        ])->count();
        $esteMes = Solicitud::whereMonth('created_at', now()->month)
                          ->whereYear('created_at', now()->year)
                          ->count();
        
        return [
            'total' => $total,
            'pendientes' => $pendientes,
            'en_gestion' => $enGestion,
            'cerradas' => $cerradas,
            'hoy' => $hoy,
            'esta_semana' => $estaSemana,
            'este_mes' => $esteMes,
        ];
    }
    
    private function getSolicitudesUrgentes()
    {
        return Solicitud::select('solicitudes.*')
            ->with(['tipoSolicitud', 'organismoRequeridor', 'gestor'])
            ->where('estado', '!=', 'cerrada')
            ->get()
            ->filter(function ($solicitud) {
                $limite = $solicitud->fecha_asignacion ?? $solicitud->created_at;
                $horasLimite = $solicitud->tiempo_limite_horas ?? 168;
                $fechaLimite = Carbon::parse($limite)->addHours($horasLimite);
                $diasRestantes = now()->diffInDays($fechaLimite, false);
                
                // Considerar urgente si quedan menos de 2 días o ya se venció
                return $diasRestantes <= 2 || $fechaLimite->isPast();
            })
            ->sortBy(function ($solicitud) {
                $limite = $solicitud->fecha_asignacion ?? $solicitud->created_at;
                $horasLimite = $solicitud->tiempo_limite_horas ?? 168;
                $fechaLimite = Carbon::parse($limite)->addHours($horasLimite);
                return now()->diffInMinutes($fechaLimite, false);
            })
            ->take(10);
    }
    
    private function getTopGestores()
    {
        return User::select('users.id', 'users.name', 'users.email', 'users.role')
            ->selectRaw('COUNT(solicitudes.id) as total_gestiones')
            ->selectRaw('COUNT(CASE WHEN solicitudes.estado IN ("pendiente", "en_gestion") THEN 1 END) as activas')
            ->selectRaw('COUNT(CASE WHEN solicitudes.estado = "cerrada" THEN 1 END) as cerradas')
            ->leftJoin('solicitudes', 'users.id', '=', 'solicitudes.gestor_id')
            ->whereNotNull('solicitudes.gestor_id')  // Solo usuarios que tienen solicitudes asignadas
            ->groupBy('users.id', 'users.name', 'users.email', 'users.role')
            ->having('total_gestiones', '>', 0)
            ->orderByDesc('cerradas')  // Primero por cerradas (más importante)
            ->orderByDesc('activas')   // Luego por activas
            ->orderByDesc('total_gestiones')  // Finalmente por total
            ->take(5)
            ->get();
    }
    
    private function getUltimasSolicitudes()
    {
        return Solicitud::with(['tipoSolicitud', 'organismoRequeridor', 'gestor'])
            ->orderByDesc('created_at')
            ->take(8)
            ->get();
    }
    
    private function getEstadisticasPorEstado()
    {
        return Solicitud::select('estado')
            ->selectRaw('COUNT(*) as total')
            ->groupBy('estado')
            ->get()
            ->pluck('total', 'estado')
            ->toArray();
    }
    
    private function getTiposSolicitudComunes()
    {
        return TipoSolicitud::withCount('solicitudes')
            ->orderByDesc('solicitudes_count')
            ->get()
            ->filter(function ($tipo) {
                return $tipo->solicitudes_count > 0;
            })
            ->take(5)
            ->map(function ($tipo) {
                $tipo->total_solicitudes = $tipo->solicitudes_count;
                return $tipo;
            });
    }
    
    private function getOrganismosActivos()
    {
        return Organismo::select('organismos.*')
            ->selectRaw('COUNT(solicitudes.id) as total_solicitudes')
            ->selectRaw('COUNT(CASE WHEN solicitudes.estado != "cerrada" THEN 1 END) as activas')
            ->leftJoin('solicitudes', 'organismos.id', '=', 'solicitudes.organismo_requeridor_id')
            ->groupBy('organismos.id', 'organismos.nombre', 'organismos.parent_id', 'organismos.created_at', 'organismos.updated_at')
            ->having('total_solicitudes', '>', 0)
            ->orderByDesc('activas')
            ->orderByDesc('total_solicitudes')
            ->take(5)
            ->get();
    }
    
    private function getTendenciaSolicitudes()
    {
        $fechas = [];
        $datos = [];
        
        for ($i = 6; $i >= 0; $i--) {
            $fecha = Carbon::now()->subDays($i);
            $fechas[] = $fecha->format('d/m');
            $datos[] = Solicitud::whereDate('created_at', $fecha)->count();
        }
        
        return [
            'fechas' => $fechas,
            'datos' => $datos
        ];
    }
    
    private function getTiempoPromedioResolucion()
    {
        $solicitudesCerradas = Solicitud::where('estado', 'cerrada')
            ->whereNotNull('fecha_cierre')
            ->whereNotNull('fecha_asignacion')
            ->get();
            
        if ($solicitudesCerradas->count() === 0) {
            return 0;
        }
        
        $tiempoTotal = 0;
        foreach ($solicitudesCerradas as $solicitud) {
            $inicio = Carbon::parse($solicitud->fecha_asignacion);
            $fin = Carbon::parse($solicitud->fecha_cierre);
            $tiempoTotal += $inicio->diffInHours($fin);
        }
        
        return round($tiempoTotal / $solicitudesCerradas->count() / 24, 1); // En días
    }
}
