<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Solicitud;
use App\Services\NotificationService;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Storage;
use Symfony\Component\HttpFoundation\Response;

class SolicitudController extends Controller
{
    public function index(Request $request)
    {
        $query = \App\Models\Solicitud::with(['tipoSolicitud', 'organismoRequeridor', 'gestor', 'gestiones']);
        
        // Filtrar por estado si se especifica
        if ($request->filled('estado')) {
            $query->where('estado', $request->estado);
        }
        
        // Ordenamiento inteligente: cerradas al final, luego por fecha según filtro
        $order = $request->get('order', 'recientes');
        
        $query->orderByRaw("
            CASE 
                WHEN estado = 'cerrada' THEN 1 
                ELSE 0 
            END ASC
        ");
        
        // Para solicitudes no cerradas, aplicar el orden seleccionado
        if ($order === 'antiguas') {
            $query->orderBy('created_at', 'asc');
        } else {
            $query->orderBy('created_at', 'desc');
        }
        
        // Paginación de 20 elementos
        $solicitudes = $query->paginate(20);
        
        // Cargar evidencias para cada solicitud
        $solicitudesConEvidencias = $solicitudes->map(function ($solicitud) {
            $solicitud->evidencias = $this->getEvidencias($solicitud->id);
            return $solicitud;
        });
        
        return view('admin.solicitudes.index', compact('solicitudes'));
    }

    public function gestionar($id)
    {
        $solicitud = Solicitud::findOrFail($id);
        
        // Verificar que la solicitud no esté cerrada
        if ($solicitud->estado === 'cerrada') {
            return redirect()->back()->with('error', 'No se puede gestionar una solicitud cerrada.');
        }
        
        // Verificar que no tenga gestor asignado
        if ($solicitud->gestor_id) {
            $gestorActual = $solicitud->gestor->name ?? 'Usuario desconocido';
            return redirect()->back()->with('error', "Esta solicitud ya está siendo gestionada por {$gestorActual}.");
        }
        
        // Asignar al usuario actual como gestor
        $solicitud->gestor_id = Auth::id();
        $solicitud->fecha_asignacion = now();
        $solicitud->estado = 'en_gestion';
        $solicitud->save();
        
        return redirect()->route('admin.solicitudes.show', $solicitud->id)->with('success', 'Ahora está gestionando esta solicitud.');
    }

    public function show($id)
    {
        $solicitud = Solicitud::with(['tipoSolicitud', 'organismoRequeridor', 'gestor', 'gestiones'])->findOrFail($id);
        $evidencias = $this->getEvidencias($id);
        return view('admin.solicitudes.show', compact('solicitud', 'evidencias'));
    }

    public function cerrar(Request $request, $id, NotificationService $notificationService)
    {
        $request->validate([
            'observaciones_cierre' => 'required|string|min:10|max:1000',
        ], [
            'observaciones_cierre.required' => 'Las observaciones de cierre son obligatorias.',
            'observaciones_cierre.min' => 'Las observaciones deben tener al menos 10 caracteres.',
            'observaciones_cierre.max' => 'Las observaciones no pueden exceder 1000 caracteres.',
        ]);

        $solicitud = Solicitud::findOrFail($id);
        
        // Verificar que la solicitud puede ser cerrada
        if ($solicitud->estado === 'cerrada') {
            return redirect()->back()->with('error', 'Esta solicitud ya está cerrada.');
        }
        
        // VALIDACIÓN DE SEGURIDAD: Solo el gestor asignado puede cerrar la solicitud
        if ($solicitud->gestor_id !== auth()->id()) {
            return redirect()->back()->with('error', 'Solo el gestor asignado puede cerrar esta solicitud.');
        }

        // Actualizar la solicitud
        $solicitud->estado = 'cerrada';
        $solicitud->fecha_cierre = now();
        $solicitud->observaciones_cierre = $request->observaciones_cierre;
        
        $solicitud->save();

        // Enviar notificación por correo al solicitante
        try {
            $notificationService->notificarCierreSolicitud($solicitud, Auth::user());
            $mensaje = 'Solicitud cerrada correctamente. Se ha enviado una notificación al solicitante.';
        } catch (\Exception $e) {
            Log::error('Error al enviar notificación de cierre: ' . $e->getMessage());
            $mensaje = 'Solicitud cerrada correctamente, pero hubo un problema al enviar la notificación por correo.';
        }

        return redirect()->route('admin.solicitudes.index')->with('success', $mensaje);
    }

    public function storeGestion(Request $request, $id)
    {
        $solicitud = \App\Models\Solicitud::findOrFail($id);
        $request->validate([
            'descripcion' => 'required|string|max:1000',
        ]);
        $solicitud->gestiones()->create([
            'descripcion_gestion' => $request->descripcion,
            'fecha_gestion' => now(),
            'accion_realizada' => $request->accion_realizada ?? $request->descripcion,
        ]);
        return redirect()->route('admin.solicitudes.show', $solicitud->id)->with('success', 'Acción registrada correctamente.');
    }

    /**
     * Obtener la lista de archivos de evidencia para una solicitud
     */
    public function getEvidencias($id)
    {
        $solicitud = Solicitud::findOrFail($id);

        if (!$solicitud->uri_evidencia || !file_exists($solicitud->uri_evidencia)) {
            return [];
        }

        $files = [];
        $evidenciaPath = $solicitud->uri_evidencia;

        if (is_dir($evidenciaPath)) {
            $scanFiles = scandir($evidenciaPath);
            foreach ($scanFiles as $file) {
                if ($file !== '.' && $file !== '..' && is_file($evidenciaPath . '/' . $file)) {
                    $filePath = $evidenciaPath . '/' . $file;
                    $files[] = [
                        'name' => $file,
                        'path' => $filePath,
                        'size' => filesize($filePath),
                        'modified' => date('d/m/Y H:i', filemtime($filePath)),
                        'download_url' => route('admin.solicitudes.download-evidencia', ['id' => $id, 'filename' => $file])
                    ];
                }
            }
        }

        return $files;
    }

    /**
     * Descargar un archivo de evidencia específico
     */
    public function downloadEvidencia($id, $filename)
    {
        $solicitud = Solicitud::findOrFail($id);

        if (!$solicitud->uri_evidencia || !file_exists($solicitud->uri_evidencia)) {
            abort(404, 'Carpeta de evidencias no encontrada');
        }

        $filePath = $solicitud->uri_evidencia . '/' . $filename;

        // Validar que el archivo existe y está dentro de la carpeta de evidencias
        if (!file_exists($filePath) || !str_starts_with(realpath($filePath), realpath($solicitud->uri_evidencia))) {
            abort(404, 'Archivo no encontrado');
        }

        // Determinar el tipo MIME del archivo
        $mimeType = mime_content_type($filePath);

        return response()->download($filePath, $filename, [
            'Content-Type' => $mimeType,
        ]);
    }

    /**
     * Descargar todas las evidencias en un archivo ZIP
     */
    public function downloadAllEvidencias($id)
    {
        $solicitud = Solicitud::findOrFail($id);

        if (!$solicitud->uri_evidencia || !file_exists($solicitud->uri_evidencia)) {
            abort(404, 'Carpeta de evidencias no encontrada');
        }

        $evidenciaPath = $solicitud->uri_evidencia;
        $zipFileName = "evidencias_solicitud_{$id}.zip";
        $zipPath = storage_path('app/temp/' . $zipFileName);

        // Crear directorio temporal si no existe
        if (!file_exists(dirname($zipPath))) {
            mkdir(dirname($zipPath), 0755, true);
        }

        $zip = new \ZipArchive();

        if ($zip->open($zipPath, \ZipArchive::CREATE | \ZipArchive::OVERWRITE) === TRUE) {
            $files = scandir($evidenciaPath);
            foreach ($files as $file) {
                if ($file !== '.' && $file !== '..' && is_file($evidenciaPath . '/' . $file)) {
                    $zip->addFile($evidenciaPath . '/' . $file, $file);
                }
            }
            $zip->close();

            return response()->download($zipPath)->deleteFileAfterSend(true);
        } else {
            abort(500, 'No se pudo crear el archivo ZIP');
        }
    }
}
