// Clase para manejar modales dinámicos
class DynamicModal {
    constructor() {
        this.modal = document.getElementById('dynamicModal');
        this.modalTitle = document.getElementById('dynamicModalLabel');
        this.modalContent = document.getElementById('modalContent');
        this.modalForm = document.getElementById('dynamicForm');
        this.submitBtn = document.getElementById('submitBtn');
        this.methodField = document.getElementById('methodField');

        this.initEventListeners();
    }

    initEventListeners() {
        // Limpiar modal al cerrarse
        $(this.modal).on('hidden.bs.modal', () => {
            this.clearModal();
        });

        // Manejar envío del formulario
        this.modalForm.addEventListener('submit', (e) => {
            this.handleFormSubmit(e);
        });
    }    // Abrir modal para ver detalles (solo lectura)
    openViewModal(title, content) {
        this.modalTitle.textContent = title;
        this.modalContent.innerHTML = content;
        this.submitBtn.style.display = 'none';
        this.modalForm.action = '#';
        this.methodField.value = '';
        
        $(this.modal).modal('show');
    }

    // Abrir modal para editar
    openEditModal(title, content, formAction, method = 'PUT') {
        this.modalTitle.textContent = title;
        this.modalContent.innerHTML = content;
        this.submitBtn.style.display = 'block';
        this.modalForm.action = formAction;
        this.methodField.value = method;
        
        $(this.modal).modal('show');
    }

    // Limpiar contenido del modal
    clearModal() {
        this.modalContent.innerHTML = '';
        this.modalForm.action = '#';
        this.methodField.value = '';
        this.submitBtn.style.display = 'none';
    }

    // Manejar envío del formulario
    handleFormSubmit(e) {
        e.preventDefault();

        const formData = new FormData(this.modalForm);
        const submitButton = this.submitBtn;
        const originalText = submitButton.textContent;

        // Deshabilitar botón durante el envío
        submitButton.disabled = true;
        submitButton.textContent = 'Guardando...';

        fetch(this.modalForm.action, {
            method: 'POST',
            body: formData,
            headers: {
                'X-Requested-With': 'XMLHttpRequest',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            }
        })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    // Cerrar modal
                    $(this.modal).modal('hide');

                    // Mostrar mensaje de éxito
                    if (typeof showToast !== 'undefined') {
                        showToast({
                            icon: 'success',
                            title: data.message || 'Operación exitosa'
                        });
                    }

                    // Recargar la página o actualizar la tabla
                    setTimeout(() => {
                        window.location.reload();
                    }, 1000);
                } else {
                    // Mostrar errores
                    if (typeof showToast !== 'undefined') {
                        showToast({
                            icon: 'error',
                            title: 'Error',
                            text: data.message || 'Ocurrió un error',
                            timer: 4000
                        });
                    }

                    // Mostrar errores de validación si existen
                    if (data.errors) {
                        this.displayValidationErrors(data.errors);
                    }
                }
            })
            .catch(error => {
                console.error('Error:', error);
                if (typeof showToast !== 'undefined') {
                    showToast({
                        icon: 'error',
                        title: 'Error',
                        text: 'Error de conexión',
                        timer: 4000
                    });
                }
            })
            .finally(() => {
                // Restaurar botón
                submitButton.disabled = false;
                submitButton.textContent = originalText;
            });
    }

    // Mostrar errores de validación
    displayValidationErrors(errors) {
        // Limpiar errores anteriores
        document.querySelectorAll('.invalid-feedback').forEach(el => el.remove());
        document.querySelectorAll('.is-invalid').forEach(el => el.classList.remove('is-invalid'));

        // Mostrar nuevos errores
        Object.keys(errors).forEach(field => {
            const input = document.querySelector(`[name="${field}"]`);
            if (input) {
                input.classList.add('is-invalid');
                const errorDiv = document.createElement('div');
                errorDiv.className = 'invalid-feedback';
                errorDiv.textContent = errors[field][0];
                input.parentNode.appendChild(errorDiv);
            }
        });
    }
}

// Funciones globales para usar en las vistas
function openViewModal(title, content) {
    if (window.dynamicModal) {
        window.dynamicModal.openViewModal(title, content);
    }
}

function openEditModal(title, content, formAction, method = 'PUT') {
    if (window.dynamicModal) {
        window.dynamicModal.openEditModal(title, content, formAction, method);
    }
}

function loadEditModal(id) {
    fetch(`${window.tipoSolicitudesIndexRoute}/${id}/edit-ajax`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                openEditModal(
                    data.title,
                    data.content,
                    `${window.tipoSolicitudesIndexRoute}/${id}`,
                    'PUT'
                );
            }
        })
        .catch(error => console.error('Error:', error));
}

function loadViewModal(id) {
    fetch(`${window.tipoSolicitudesIndexRoute}/${id}/show-ajax`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                openViewModal(data.title, data.content);
            }
        })
        .catch(error => console.error('Error:', error));
}

document.addEventListener('DOMContentLoaded', function () {
    window.dynamicModal = new DynamicModal();
});
