/**
 * JavaScript para el dashboard del admin
 * Gráficos y funcionalidades interactivas
 */

document.addEventListener('DOMContentLoaded', function() {
    initializeDashboard();
});

/**
 * Inicializa el dashboard
 */
function initializeDashboard() {
    initializeCharts();
    initializeInteractiveElements();
    initializeDataRefresh();
}

/**
 * Inicializa los gráficos del dashboard
 */
function initializeCharts() {
    initializeTrendChart();
    initializeStatusChart();
}

/**
 * Inicializa el gráfico de tendencias
 */
function initializeTrendChart() {
    const canvas = document.getElementById('tendenciaChart');
    if (!canvas) return;

    const ctx = canvas.getContext('2d');
    
    new Chart(ctx, {
        type: 'line',
        data: {
            labels: window.dashboardData.tendenciaSolicitudes.fechas || [],
            datasets: [{
                label: 'Solicitudes',
                data: window.dashboardData.tendenciaSolicitudes.datos || [],
                borderColor: '#007bff',
                backgroundColor: 'rgba(0, 123, 255, 0.1)',
                borderWidth: 3,
                fill: true,
                tension: 0.4,
                pointBackgroundColor: '#007bff',
                pointBorderColor: '#ffffff',
                pointBorderWidth: 2,
                pointRadius: 5
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    display: false
                },
                tooltip: {
                    backgroundColor: 'rgba(0, 0, 0, 0.8)',
                    titleColor: '#ffffff',
                    bodyColor: '#ffffff',
                    borderColor: '#007bff',
                    borderWidth: 1,
                    cornerRadius: 8,
                    callbacks: {
                        title: function(context) {
                            return `Fecha: ${context[0].label}`;
                        },
                        label: function(context) {
                            return `Solicitudes: ${context.parsed.y}`;
                        }
                    }
                }
            },
            scales: {
                y: {
                    beginAtZero: true,
                    grid: {
                        color: 'rgba(0,0,0,0.1)',
                        drawBorder: false
                    },
                    ticks: {
                        color: '#6c757d',
                        font: {
                            family: 'Inter, system-ui, sans-serif'
                        }
                    }
                },
                x: {
                    grid: {
                        display: false
                    },
                    ticks: {
                        color: '#6c757d',
                        font: {
                            family: 'Inter, system-ui, sans-serif'
                        }
                    }
                }
            },
            interaction: {
                intersect: false,
                mode: 'index'
            }
        }
    });
}

/**
 * Inicializa el gráfico de estado
 */
function initializeStatusChart() {
    const canvas = document.getElementById('estadoChart');
    if (!canvas) return;

    const ctx = canvas.getContext('2d');
    const estadosData = window.dashboardData.estadisticasPorEstado || {};

    const colors = {
        'pendiente': '#ffc107',
        'en_proceso': '#17a2b8',
        'completada': '#28a745',
        'cerrada': '#6c757d',
        'urgente': '#dc3545'
    };

    const labels = Object.keys(estadosData).map(estado => {
        const labelMap = {
            'pendiente': 'Pendiente',
            'en_proceso': 'En Proceso',
            'completada': 'Completada',
            'cerrada': 'Cerrada',
            'urgente': 'Urgente'
        };
        return labelMap[estado] || estado.charAt(0).toUpperCase() + estado.slice(1).replace('_', ' ');
    });

    const backgroundColors = Object.keys(estadosData).map(estado => 
        colors[estado] || '#6c757d'
    );

    new Chart(ctx, {
        type: 'doughnut',
        data: {
            labels: labels,
            datasets: [{
                data: Object.values(estadosData),
                backgroundColor: backgroundColors,
                borderWidth: 3,
                borderColor: '#ffffff',
                hoverBorderWidth: 4,
                hoverOffset: 10
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    position: 'bottom',
                    labels: {
                        padding: 20,
                        usePointStyle: true,
                        font: {
                            family: 'Inter, system-ui, sans-serif',
                            size: 12
                        },
                        color: '#495057'
                    }
                },
                tooltip: {
                    backgroundColor: 'rgba(0, 0, 0, 0.8)',
                    titleColor: '#ffffff',
                    bodyColor: '#ffffff',
                    borderColor: '#007bff',
                    borderWidth: 1,
                    cornerRadius: 8,
                    callbacks: {
                        label: function(context) {
                            const total = context.dataset.data.reduce((a, b) => a + b, 0);
                            const percentage = ((context.parsed / total) * 100).toFixed(1);
                            return `${context.label}: ${context.parsed} (${percentage}%)`;
                        }
                    }
                }
            },
            cutout: '60%',
            animation: {
                animateScale: true,
                animateRotate: true
            }
        }
    });
}

/**
 * Inicializa elementos interactivos
 */
function initializeInteractiveElements() {
    initializeStatsCardAnimations();
    initializeTableInteractions();
    initializeQuickActions();
}

/**
 * Inicializa animaciones de las tarjetas de estadísticas
 */
function initializeStatsCardAnimations() {
    const statsNumbers = document.querySelectorAll('.stats-number');
    
    statsNumbers.forEach((element, index) => {
        const finalValue = parseInt(element.textContent) || 0;
        element.textContent = '0';
        
        // Animar contador con delay escalonado
        setTimeout(() => {
            animateCounter(element, 0, finalValue, 1500);
        }, index * 200);
    });
}

/**
 * Anima un contador numérico
 * @param {HTMLElement} element - Elemento a animar
 * @param {number} start - Valor inicial
 * @param {number} end - Valor final
 * @param {number} duration - Duración en ms
 */
function animateCounter(element, start, end, duration) {
    const startTime = performance.now();
    
    function update(currentTime) {
        const elapsed = currentTime - startTime;
        const progress = Math.min(elapsed / duration, 1);
        
        // Ease out cubic para suavizar la animación
        const easeProgress = 1 - Math.pow(1 - progress, 3);
        
        const currentValue = Math.round(start + (end - start) * easeProgress);
        element.textContent = currentValue.toLocaleString();
        
        if (progress < 1) {
            requestAnimationFrame(update);
        }
    }
    
    requestAnimationFrame(update);
}

/**
 * Inicializa interacciones de tabla
 */
function initializeTableInteractions() {
    const tableRows = document.querySelectorAll('.dashboard-table tbody tr');
    
    tableRows.forEach(row => {
        row.addEventListener('click', function() {
            const solicitudId = this.dataset.solicitudId;
            if (solicitudId) {
                // Aquí se puede agregar lógica para mostrar detalles
                console.log('Clicked solicitud:', solicitudId);
            }
        });
    });
}

/**
 * Inicializa acciones rápidas
 */
function initializeQuickActions() {
    const quickActions = document.querySelectorAll('.quick-action');
    
    quickActions.forEach(action => {
        action.addEventListener('click', function(e) {
            // Agregar efecto de clic
            this.style.transform = 'scale(0.95)';
            setTimeout(() => {
                this.style.transform = '';
            }, 150);
        });
    });
}

/**
 * Inicializa el refresco automático de datos
 */
function initializeDataRefresh() {
    // Refrescar cada 5 minutos
    setInterval(() => {
        refreshDashboardData();
    }, 5 * 60 * 1000);
}

/**
 * Refresca los datos del dashboard
 */
function refreshDashboardData() {
    fetch('/admin/dashboard/refresh', {
        method: 'GET',
        headers: {
            'X-Requested-With': 'XMLHttpRequest',
            'Accept': 'application/json'
        }
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            updateDashboardStats(data.estadisticas);
            showRefreshNotification();
        }
    })
    .catch(error => {
        console.error('Error refreshing dashboard:', error);
    });
}

/**
 * Actualiza las estadísticas del dashboard
 * @param {Object} stats - Nuevas estadísticas
 */
function updateDashboardStats(stats) {
    const statsElements = {
        total: document.querySelector('.stats-card.consolidated .stats-number'),
        pendientes: document.querySelector('.stats-card.warning .stats-number'),
        completadas: document.querySelector('.stats-card.success .stats-number'),
        cerradas: document.querySelector('.stats-card.info .stats-number')
    };
    
    Object.keys(statsElements).forEach(key => {
        const element = statsElements[key];
        if (element && stats[key] !== undefined) {
            const currentValue = parseInt(element.textContent.replace(/,/g, ''));
            const newValue = stats[key];
            
            if (currentValue !== newValue) {
                animateCounter(element, currentValue, newValue, 1000);
            }
        }
    });
}

/**
 * Muestra notificación de refresco
 */
function showRefreshNotification() {
    const notification = document.createElement('div');
    notification.className = 'refresh-notification';
    notification.innerHTML = '<i class="fas fa-sync-alt"></i> Datos actualizados';
    notification.style.cssText = `
        position: fixed;
        top: 20px;
        right: 20px;
        background: #28a745;
        color: white;
        padding: 0.75rem 1rem;
        border-radius: 0.375rem;
        font-size: 0.875rem;
        z-index: 1000;
        opacity: 0;
        transform: translateX(100%);
        transition: all 0.3s ease;
    `;
    
    document.body.appendChild(notification);
    
    // Animar entrada
    setTimeout(() => {
        notification.style.opacity = '1';
        notification.style.transform = 'translateX(0)';
    }, 100);
    
    // Animar salida y remover
    setTimeout(() => {
        notification.style.opacity = '0';
        notification.style.transform = 'translateX(100%)';
        setTimeout(() => {
            document.body.removeChild(notification);
        }, 300);
    }, 3000);
}
