/**
 * JavaScript para formularios de organismos en el admin
 * Funcionalidad de jerarquía y validación
 */

document.addEventListener('DOMContentLoaded', function() {
    initializeOrganismoForm();
});

/**
 * Inicializa el formulario de organismos
 */
function initializeOrganismoForm() {
    updatePath(); // Mostrar path inicial
    
    // Agregar listeners a los selectores de nivel
    for (let i = 1; i <= 4; i++) {
        const select = document.getElementById(`level${i}`);
        if (select) {
            select.addEventListener('change', function() {
                updateHierarchy(i);
            });
        }
    }
}

/**
 * Actualiza la jerarquía cuando se selecciona un nivel
 * @param {number} level - Nivel seleccionado
 */
function updateHierarchy(level) {
    const currentSelect = document.getElementById(`level${level}`);
    const selectedValue = currentSelect.value;
    const selectedName = currentSelect.options[currentSelect.selectedIndex]?.getAttribute('data-name') || '';
    
    // Limpiar niveles siguientes
    for (let i = level + 1; i <= 4; i++) {
        const nextSelect = document.getElementById(`level${i}`);
        nextSelect.innerHTML = '<option value="">Seleccionar...</option>';
        nextSelect.disabled = true;
        nextSelect.classList.add('disabled-select');
    }
    
    if (selectedValue && level < 4) {
        const nextLevel = level + 1;
        const nextSelect = document.getElementById(`level${nextLevel}`);
        
        loadChildren(selectedValue, nextSelect);
        
        nextSelect.disabled = false;
        nextSelect.classList.remove('disabled-select');
    }
    
    // Actualizar el path y el parent_id
    updatePath();
    updateParentId();
}

/**
 * Carga los hijos de un organismo
 * @param {string} parentId - ID del organismo padre
 * @param {HTMLSelectElement} selectElement - Elemento select destino
 */
function loadChildren(parentId, selectElement) {
    fetch(`/admin/organismos/${parentId}/children-ajax`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                selectElement.innerHTML = '<option value="">Crear organismo en este nivel</option>';
                data.children.forEach(child => {
                    const option = document.createElement('option');
                    option.value = child.id;
                    option.setAttribute('data-name', child.nombre);
                    option.textContent = child.nombre;
                    selectElement.appendChild(option);
                });
            }
        })
        .catch(error => {
            console.error('Error cargando organismos dependientes:', error);
            if (typeof Swal !== 'undefined') {
                Swal.fire({
                    icon: 'error',
                    title: 'Error',
                    text: 'No se pudieron cargar los organismos dependientes'
                });
            }
        });
}

/**
 * Actualiza el path visual de la jerarquía
 */
function updatePath() {
    let path = [];
    let currentLevel = 1;
    
    while (currentLevel <= 4) {
        const select = document.getElementById(`level${currentLevel}`);
        const selectedOption = select.options[select.selectedIndex];
        
        if (select.value) {
            const name = selectedOption.getAttribute('data-name') || selectedOption.textContent;
            path.push(`Nivel ${currentLevel}: ${name}`);
            currentLevel++;
        } else {
            break;
        }
    }
    
    const pathText = document.getElementById('pathText');
    if (path.length === 0) {
        pathText.textContent = 'Nivel raíz (sin organismo padre)';
    } else {
        const nextLevel = path.length + 1;
        pathText.innerHTML = path.join(' → ') + ` → <strong>Nuevo organismo (Nivel ${nextLevel})</strong>`;
    }
}

/**
 * Actualiza el parent_id del formulario
 */
function updateParentId() {
    let parentId = '';
    
    // Buscar el último nivel seleccionado
    for (let i = 4; i >= 1; i--) {
        const select = document.getElementById(`level${i}`);
        if (select.value) {
            parentId = select.value;
            break;
        }
    }
    
    document.getElementById('selectedParentId').value = parentId;
}

/**
 * Valida el formulario antes del envío
 * @param {Event} event - Evento del formulario
 */
function validateOrganismoForm(event) {
    const nombre = document.getElementById('nombre').value.trim();
    
    if (!nombre) {
        event.preventDefault();
        if (typeof Swal !== 'undefined') {
            Swal.fire({
                icon: 'warning',
                title: 'Campo requerido',
                text: 'El nombre del organismo es obligatorio'
            });
        } else {
            alert('El nombre del organismo es obligatorio');
        }
        return false;
    }
    
    return true;
}

// Agregar validación al formulario si existe
document.addEventListener('DOMContentLoaded', function() {
    const form = document.getElementById('organismoForm');
    if (form) {
        form.addEventListener('submit', validateOrganismoForm);
    }
});
