/**
 * =========================================
 * ADMIN SOLICITUDES - JavaScript Modular
 * =========================================
 */

// Variables globales
let currentImageModal = null;
let activeTooltips = [];

/**
 * Inicialización principal
 */
document.addEventListener('DOMContentLoaded', function() {
    initializeSolicitudes();
    showGlobalMessages();
});

/**
 * Función principal de inicialización
 */
function initializeSolicitudes() {
    initializeFilters();
    initializeActionButtons();
    initializeModals();
    initializeTooltips();
    initializeImagePreview();
    initializeEvidencias();
    initializeKeyboardShortcuts();
    initializeTableAnimations();
}

/**
 * Inicializar filtros automáticos
 */
function initializeFilters() {
    const filterSelects = document.querySelectorAll('.filtro-select');
    
    filterSelects.forEach(select => {
        select.addEventListener('change', function() {
            showLoadingInButton();
            this.closest('form').submit();
        });
    });
    
    // Animación en filtros
    const filterForm = document.querySelector('.filtros-form');
    if (filterForm) {
        filterForm.addEventListener('submit', function() {
            showLoadingInButton();
        });
    }
}

/**
 * Mostrar loading en botón de filtrar
 */
function showLoadingInButton() {
    const filterBtn = document.querySelector('.btn-filtrar');
    if (filterBtn) {
        const originalContent = filterBtn.innerHTML;
        filterBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Filtrando...';
        filterBtn.disabled = true;
        
        // Restaurar después de 3 segundos como fallback
        setTimeout(() => {
            filterBtn.innerHTML = originalContent;
            filterBtn.disabled = false;
        }, 3000);
    }
}

/**
 * Inicializar botones de acción
 */
function initializeActionButtons() {
    initializeCloseButtons();
    initializeManageButtons();
    initializeViewButtons();
}

/**
 * Botones de cerrar solicitud
 */
function initializeCloseButtons() {
    const closeButtons = document.querySelectorAll('.btn-cerrar');
    
    closeButtons.forEach(button => {
        button.addEventListener('click', function(e) {
            e.preventDefault();
            
            const solicitudId = this.dataset.solicitudId;
            confirmCloseSolicitud(solicitudId, this.closest('form'));
        });
    });
}

/**
 * Confirmar cierre de solicitud
 */
function confirmCloseSolicitud(solicitudId, form) {
    if (typeof Swal !== 'undefined') {
        Swal.fire({
            title: '¿Cerrar solicitud?',
            text: '¿Está seguro de que desea cerrar esta solicitud?',
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#e74c3c',
            cancelButtonColor: '#6c757d',
            confirmButtonText: 'Sí, cerrar',
            cancelButtonText: 'Cancelar',
            reverseButtons: true
        }).then((result) => {
            if (result.isConfirmed) {
                showLoadingAndSubmit(form, 'Cerrando solicitud...');
            }
        });
    } else {
        // Fallback para confirmación nativa
        if (confirm('¿Está seguro de cerrar esta solicitud?')) {
            showLoadingAndSubmit(form, 'Cerrando solicitud...');
        }
    }
}

/**
 * Botones de gestionar solicitud
 */
function initializeManageButtons() {
    const manageButtons = document.querySelectorAll('.btn-gestionar');
    
    manageButtons.forEach(button => {
        button.addEventListener('click', function(e) {
            e.preventDefault();
            showLoadingAndSubmit(this.closest('form'), 'Iniciando gestión...');
        });
    });
}

/**
 * Botones de ver detalle
 */
function initializeViewButtons() {
    const viewButtons = document.querySelectorAll('.btn-ver');
    
    viewButtons.forEach(button => {
        button.addEventListener('click', function() {
            const modalTarget = this.dataset.target;
            prepareModalLoading(modalTarget);
        });
    });
}

/**
 * Preparar loading en modal
 */
function prepareModalLoading(modalTarget) {
    const modal = document.querySelector(modalTarget);
    if (modal) {
        const modalBody = modal.querySelector('.modal-body');
        if (modalBody) {
            addLoadingOverlay(modalBody);
        }
    }
}

/**
 * Mostrar loading y enviar formulario
 */
function showLoadingAndSubmit(form, message = 'Procesando...') {
    const submitButton = form.querySelector('button[type="submit"]');
    if (submitButton) {
        const originalContent = submitButton.innerHTML;
        submitButton.innerHTML = `<i class="fas fa-spinner fa-spin"></i> ${message}`;
        submitButton.disabled = true;
    }
    
    form.submit();
}

/**
 * Agregar overlay de loading
 */
function addLoadingOverlay(element) {
    const overlay = document.createElement('div');
    overlay.className = 'loading-overlay d-flex justify-content-center align-items-center';
    overlay.style.cssText = `
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        bottom: 0;
        background: rgba(255,255,255,0.9);
        z-index: 1000;
        border-radius: 8px;
    `;
    
    overlay.innerHTML = `
        <div class="text-center">
            <div class="spinner-border text-primary mb-2" role="status">
                <span class="sr-only">Cargando...</span>
            </div>
            <div class="text-muted">Cargando información...</div>
        </div>
    `;
    
    element.style.position = 'relative';
    element.appendChild(overlay);
    
    // Remover loading después de un tiempo
    setTimeout(() => {
        overlay.style.opacity = '0';
        setTimeout(() => {
            if (overlay.parentNode) {
                overlay.parentNode.removeChild(overlay);
            }
        }, 300);
    }, 800);
}

/**
 * Inicializar modales
 */
function initializeModals() {
    // Mejorar animaciones de modales
    const modals = document.querySelectorAll('.modal');
    
    modals.forEach(modal => {
        modal.addEventListener('show.bs.modal', function() {
            const dialog = this.querySelector('.modal-dialog');
            if (dialog) {
                dialog.style.opacity = '0';
                dialog.style.transform = 'scale(0.9) translateY(-20px)';
                
                setTimeout(() => {
                    dialog.style.transition = 'all 0.3s ease';
                    dialog.style.opacity = '1';
                    dialog.style.transform = 'scale(1) translateY(0)';
                }, 10);
            }
        });
        
        modal.addEventListener('hidden.bs.modal', function() {
            // Limpiar cualquier loading overlay
            const overlays = this.querySelectorAll('.loading-overlay');
            overlays.forEach(overlay => overlay.remove());
        });
    });
}

/**
 * Inicializar tooltips
 */
function initializeTooltips() {
    // Limpiar tooltips anteriores
    activeTooltips.forEach(tooltip => {
        if (tooltip.dispose) tooltip.dispose();
    });
    activeTooltips = [];
    
    // Inicializar nuevos tooltips
    const tooltipElements = document.querySelectorAll('[title], [data-bs-title]');
    
    tooltipElements.forEach(element => {
        if (typeof bootstrap !== 'undefined' && bootstrap.Tooltip) {
            const tooltip = new bootstrap.Tooltip(element, {
                placement: 'auto',
                trigger: 'hover focus',
                delay: { show: 500, hide: 100 }
            });
            activeTooltips.push(tooltip);
        }
    });
}

/**
 * Inicializar vista previa de imágenes
 */
function initializeImagePreview() {
    // Los botones de vista previa se manejan con onclick en el HTML
    // pero podemos agregar funcionalidad adicional aquí
    document.addEventListener('click', function(e) {
        if (e.target.closest('.btn-preview-image')) {
            e.preventDefault();
            const button = e.target.closest('.btn-preview-image');
            const imageUrl = button.dataset.imageUrl;
            const imageName = button.dataset.imageName;
            
            if (imageUrl && imageName) {
                previewImage(imageUrl, imageName);
            }
        }
    });
}

/**
 * Vista previa de imagen (función global para usar desde HTML)
 */
window.previewImage = function(imageUrl, imageName) {
    // Crear modal de vista previa
    const modalHtml = `
        <div class="modal fade image-preview-modal" id="imagePreviewModal" tabindex="-1" role="dialog" aria-labelledby="imagePreviewLabel" aria-hidden="true">
            <div class="modal-dialog modal-lg modal-dialog-centered" role="document">
                <div class="modal-content">
                    <div class="modal-header bg-dark text-white">
                        <h5 class="modal-title" id="imagePreviewLabel">
                            <i class="fas fa-image mr-2"></i>Vista previa: ${escapeHtml(imageName)}
                        </h5>
                        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal" aria-label="Cerrar"></button>
                    </div>
                    <div class="modal-body text-center p-0">
                        <div class="loading-image d-flex justify-content-center align-items-center" id="imageLoading" style="min-height: 200px;">
                            <div class="text-center">
                                <div class="spinner-border text-primary mb-2" role="status">
                                    <span class="sr-only">Cargando...</span>
                                </div>
                                <div class="text-muted">Cargando imagen...</div>
                            </div>
                        </div>
                        <img id="previewImage" src="" class="img-fluid d-none" alt="${escapeHtml(imageName)}" style="max-height: 70vh; border-radius: 0 0 8px 8px;">
                    </div>
                    <div class="modal-footer bg-light">
                        <a href="${escapeHtml(imageUrl)}" class="btn btn-primary" download="${escapeHtml(imageName)}">
                            <i class="fas fa-download mr-1"></i>Descargar
                        </a>
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">
                            <i class="fas fa-times mr-1"></i>Cerrar
                        </button>
                    </div>
                </div>
            </div>
        </div>
    `;
    
    // Remover modal anterior si existe
    const existingModal = document.getElementById('imagePreviewModal');
    if (existingModal) {
        existingModal.remove();
    }
    
    // Agregar nuevo modal al body
    document.body.insertAdjacentHTML('beforeend', modalHtml);
    
    // Mostrar modal
    const modal = document.getElementById('imagePreviewModal');
    const bsModal = new bootstrap.Modal(modal);
    bsModal.show();
    
    // Cargar imagen
    const img = document.getElementById('previewImage');
    const loading = document.getElementById('imageLoading');
    
    img.onload = function() {
        loading.classList.add('d-none');
        img.classList.remove('d-none');
    };
    
    img.onerror = function() {
        loading.innerHTML = `
            <div class="text-center text-danger">
                <i class="fas fa-exclamation-triangle fa-2x mb-2"></i>
                <div>Error al cargar la imagen</div>
            </div>
        `;
    };
    
    img.src = imageUrl;
    
    // Limpiar modal cuando se cierre
    modal.addEventListener('hidden.bs.modal', function() {
        modal.remove();
    });
    
    currentImageModal = bsModal;
};

/**
 * =========================================
 * GESTIÓN DE EVIDENCIAS
 * =========================================
 */

/**
 * Inicializar funcionalidades de evidencias
 */
function initializeEvidencias() {
    initializeDownloadAll();
    initializeCopyLinks();
    initializeEvidenciaPreview();
}

/**
 * Inicializar descarga múltiple de evidencias
 */
function initializeDownloadAll() {
    const downloadAllBtns = document.querySelectorAll('.download-all-btn');
    
    downloadAllBtns.forEach(btn => {
        btn.addEventListener('click', function() {
            const solicitudId = this.dataset.solicitudId;
            if (solicitudId) {
                downloadAllEvidences(solicitudId);
            }
        });
    });
}

/**
 * Descargar todas las evidencias de una solicitud
 */
function downloadAllEvidences(solicitudId) {
    const btn = document.querySelector(`[data-solicitud-id="${solicitudId}"]`);
    
    if (btn) {
        // Mostrar loading
        const originalContent = btn.innerHTML;
        btn.innerHTML = '<i class="fas fa-spinner fa-spin me-1"></i>Preparando descarga...';
        btn.disabled = true;
        
        // Simular descarga (aquí se implementaría la descarga real)
        setTimeout(() => {
            // Restaurar botón
            btn.innerHTML = originalContent;
            btn.disabled = false;
            
            // Mostrar notificación
            showNotification('Descarga iniciada', 'Las evidencias se están descargando...', 'success');
            
            // Aquí iría la lógica real de descarga
            // window.location.href = `/admin/solicitudes/${solicitudId}/download-all-evidencias`;
        }, 1500);
    }
}

/**
 * Inicializar copiar enlaces
 */
function initializeCopyLinks() {
    const copyBtns = document.querySelectorAll('.copy-link-btn');
    
    copyBtns.forEach(btn => {
        btn.addEventListener('click', function() {
            const url = this.dataset.url;
            if (url) {
                copyToClipboard(url, this);
            }
        });
    });
}

/**
 * Copiar enlace al portapapeles
 */
function copyToClipboard(text, button) {
    if (navigator.clipboard && window.isSecureContext) {
        // Usar API moderna
        navigator.clipboard.writeText(text).then(() => {
            showCopySuccess(button);
        }).catch(() => {
            fallbackCopyTextToClipboard(text, button);
        });
    } else {
        // Fallback para navegadores antiguos
        fallbackCopyTextToClipboard(text, button);
    }
}

/**
 * Método fallback para copiar texto
 */
function fallbackCopyTextToClipboard(text, button) {
    const textArea = document.createElement('textarea');
    textArea.value = text;
    textArea.style.position = 'fixed';
    textArea.style.left = '-999999px';
    textArea.style.top = '-999999px';
    document.body.appendChild(textArea);
    textArea.focus();
    textArea.select();
    
    try {
        document.execCommand('copy');
        showCopySuccess(button);
    } catch (err) {
        console.error('Error al copiar: ', err);
        showNotification('Error', 'No se pudo copiar el enlace', 'error');
    }
    
    document.body.removeChild(textArea);
}

/**
 * Mostrar éxito al copiar
 */
function showCopySuccess(button) {
    const originalContent = button.innerHTML;
    const originalClass = button.className;
    
    // Cambiar a estado de éxito
    button.innerHTML = '<i class="fas fa-check me-1"></i>Copiado';
    button.className = button.className.replace('btn-outline-secondary', 'btn-success');
    button.disabled = true;
    
    // Restaurar después de 2 segundos
    setTimeout(() => {
        button.innerHTML = originalContent;
        button.className = originalClass;
        button.disabled = false;
    }, 2000);
    
    // Mostrar notificación
    showNotification('Enlace copiado', 'El enlace se ha copiado al portapapeles', 'success');
}

/**
 * Inicializar vista previa de evidencias (imágenes)
 */
function initializeEvidenciaPreview() {
    const previewBtns = document.querySelectorAll('.preview-btn');
    
    previewBtns.forEach(btn => {
        btn.addEventListener('click', function() {
            const imageUrl = this.dataset.imageUrl;
            const imageName = this.dataset.imageName;
            
            if (imageUrl && imageName) {
                showImagePreview(imageUrl, imageName);
            }
        });
    });
}

/**
 * Mostrar vista previa de imagen
 */
function showImagePreview(imageUrl, imageName) {
    // Crear modal si no existe
    let modal = document.getElementById('imagePreviewModal');
    if (!modal) {
        modal = createImagePreviewModal();
    }
    
    // Actualizar contenido
    const modalImg = modal.querySelector('.preview-image');
    const modalTitle = modal.querySelector('.modal-title');
    const downloadLink = modal.querySelector('.download-link');
    
    modalImg.src = imageUrl;
    modalImg.alt = imageName;
    modalTitle.textContent = imageName;
    downloadLink.href = imageUrl;
    downloadLink.download = imageName;
    
    // Mostrar modal
    const bootstrapModal = new bootstrap.Modal(modal);
    bootstrapModal.show();
    
    currentImageModal = bootstrapModal;
}

/**
 * Crear modal de vista previa de imagen
 */
function createImagePreviewModal() {
    const modalHtml = `
        <div class="modal fade" id="imagePreviewModal" tabindex="-1" aria-labelledby="imagePreviewModalLabel" aria-hidden="true">
            <div class="modal-dialog modal-lg modal-dialog-centered">
                <div class="modal-content">
                    <div class="modal-header">
                        <h5 class="modal-title" id="imagePreviewModalLabel">Vista previa</h5>
                        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Cerrar"></button>
                    </div>
                    <div class="modal-body text-center">
                        <img src="" alt="" class="preview-image img-fluid rounded" style="max-height: 70vh;">
                    </div>
                    <div class="modal-footer">
                        <a href="" class="btn btn-primary download-link" download>
                            <i class="fas fa-download me-1"></i>Descargar imagen
                        </a>
                        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cerrar</button>
                    </div>
                </div>
            </div>
        </div>
    `;
    
    document.body.insertAdjacentHTML('beforeend', modalHtml);
    return document.getElementById('imagePreviewModal');
}

/**
 * Inicializar atajos de teclado
 */
function initializeKeyboardShortcuts() {
    document.addEventListener('keydown', function(e) {
        // Cerrar modal de imagen con Escape
        if (e.key === 'Escape' && currentImageModal) {
            currentImageModal.hide();
        }
        
        // Descargar imagen con Ctrl+S
        if (e.ctrlKey && e.key === 's' && currentImageModal) {
            e.preventDefault();
            const downloadLink = document.querySelector('#imagePreviewModal a[download]');
            if (downloadLink) {
                downloadLink.click();
            }
        }
        
        // Actualizar página con F5 o Ctrl+R
        if (e.key === 'F5' || (e.ctrlKey && e.key === 'r')) {
            // Permitir recarga normal
        }
    });
}

/**
 * Inicializar animaciones de tabla
 */
function initializeTableAnimations() {
    const tableRows = document.querySelectorAll('.solicitudes-table tbody tr');
    
    // Agregar efecto de hover mejorado
    tableRows.forEach(row => {
        row.addEventListener('mouseenter', function() {
            this.style.transform = 'scale(1.01)';
            this.style.zIndex = '10';
        });
        
        row.addEventListener('mouseleave', function() {
            this.style.transform = 'scale(1)';
            this.style.zIndex = 'auto';
        });
    });
}

/**
 * Mostrar mensajes globales
 */
function showGlobalMessages() {
    if (window.successMessage) {
        showToast('success', 'Éxito', window.successMessage);
    }
    
    if (window.errorMessage) {
        showToast('error', 'Error', window.errorMessage);
    }
    
    if (window.errorMessages) {
        showToast('error', 'Errores encontrados', window.errorMessages);
    }
}

/**
 * Mostrar toast notification
 */
function showToast(type, title, message) {
    if (typeof Swal !== 'undefined') {
        Swal.fire({
            icon: type,
            title: title,
            html: message,
            timer: 4000,
            showConfirmButton: false,
            toast: true,
            position: 'top-end',
            timerProgressBar: true,
            didOpen: (toast) => {
                toast.addEventListener('mouseenter', Swal.stopTimer);
                toast.addEventListener('mouseleave', Swal.resumeTimer);
            }
        });
    }
}

/**
 * Escapar HTML para prevenir XSS
 */
function escapeHtml(text) {
    const map = {
        '&': '&amp;',
        '<': '&lt;',
        '>': '&gt;',
        '"': '&quot;',
        "'": '&#039;'
    };
    return text.replace(/[&<>"']/g, function(m) { return map[m]; });
}

/**
 * Función para copiar enlace de descarga (uso futuro)
 */
window.copyDownloadLink = function(url) {
    navigator.clipboard.writeText(url).then(function() {
        showToast('success', '¡Enlace copiado!', 'El enlace de descarga se ha copiado al portapapeles');
    }).catch(function(err) {
        console.error('Error al copiar enlace:', err);
        showToast('error', 'Error', 'No se pudo copiar el enlace');
    });
};

/**
 * Función para descargar todas las evidencias (uso futuro)
 */
window.downloadAllEvidences = function(solicitudId) {
    if (confirm('¿Desea descargar todas las evidencias de esta solicitud?')) {
        window.location.href = `/admin/solicitudes/${solicitudId}/evidencias/download-all`;
    }
};
            e.preventDefault();
            
            const form = this.closest('form');
            const solicitudId = this.dataset.solicitudId || 'esta solicitud';



