/**
 * JavaScript para formularios de usuarios en el admin
 * Validación, fortaleza de contraseña y funcionalidades del formulario
 */

document.addEventListener('DOMContentLoaded', function() {
    initializeUserForm();
});

/**
 * Inicializa el formulario de usuarios
 */
function initializeUserForm() {
    initializePasswordValidation();
    initializeFormValidation();
    initializePasswordToggle();
    setupFormSubmission();
}

/**
 * Inicializa la validación de contraseñas
 */
function initializePasswordValidation() {
    const passwordField = document.getElementById('password');
    const confirmPasswordField = document.getElementById('password_confirmation');
    
    if (passwordField) {
        passwordField.addEventListener('input', function() {
            validatePasswordStrength(this.value);
            validatePasswordMatch();
        });
        
        // Crear indicador de fortaleza
        createPasswordStrengthIndicator(passwordField);
    }
    
    if (confirmPasswordField) {
        confirmPasswordField.addEventListener('input', validatePasswordMatch);
    }
}

/**
 * Crea el indicador de fortaleza de contraseña
 * @param {HTMLElement} passwordField - Campo de contraseña
 */
function createPasswordStrengthIndicator(passwordField) {
    const strengthContainer = document.createElement('div');
    strengthContainer.className = 'password-strength';
    
    const strengthBar = document.createElement('div');
    strengthBar.className = 'password-strength-bar';
    strengthBar.id = 'passwordStrengthBar';
    
    strengthContainer.appendChild(strengthBar);
    passwordField.parentNode.appendChild(strengthContainer);
    
    // Agregar texto de ayuda
    const helpText = document.createElement('small');
    helpText.className = 'form-text text-muted';
    helpText.textContent = 'Mínimo 8 caracteres, incluya mayúsculas, minúsculas y números';
    passwordField.parentNode.appendChild(helpText);
}

/**
 * Valida la fortaleza de la contraseña
 * @param {string} password - Contraseña a validar
 */
function validatePasswordStrength(password) {
    const strengthBar = document.getElementById('passwordStrengthBar');
    if (!strengthBar) return;
    
    let strength = 0;
    let strengthClass = '';
    
    // Criterios de fortaleza
    if (password.length >= 8) strength++;
    if (/[A-Z]/.test(password)) strength++;
    if (/[a-z]/.test(password)) strength++;
    if (/[0-9]/.test(password)) strength++;
    if (/[^A-Za-z0-9]/.test(password)) strength++;
    
    // Remover clases anteriores
    strengthBar.className = 'password-strength-bar';
    
    // Asignar nueva clase según fortaleza
    switch (strength) {
        case 0:
        case 1:
            strengthClass = 'strength-weak';
            break;
        case 2:
            strengthClass = 'strength-fair';
            break;
        case 3:
        case 4:
            strengthClass = 'strength-good';
            break;
        case 5:
            strengthClass = 'strength-strong';
            break;
    }
    
    if (password.length > 0) {
        strengthBar.classList.add(strengthClass);
    }
}

/**
 * Valida que las contraseñas coincidan
 */
function validatePasswordMatch() {
    const passwordField = document.getElementById('password');
    const confirmPasswordField = document.getElementById('password_confirmation');
    
    if (!passwordField || !confirmPasswordField) return;
    
    const password = passwordField.value;
    const confirmPassword = confirmPasswordField.value;
    
    // Limpiar estilos anteriores
    confirmPasswordField.classList.remove('is-valid', 'is-invalid');
    
    if (confirmPassword.length > 0) {
        if (password === confirmPassword) {
            confirmPasswordField.classList.add('is-valid');
            setFieldMessage(confirmPasswordField, 'Las contraseñas coinciden', 'valid');
        } else {
            confirmPasswordField.classList.add('is-invalid');
            setFieldMessage(confirmPasswordField, 'Las contraseñas no coinciden', 'invalid');
        }
    } else {
        removeFieldMessage(confirmPasswordField);
    }
}

/**
 * Inicializa la funcionalidad de mostrar/ocultar contraseña
 */
function initializePasswordToggle() {
    const passwordFields = document.querySelectorAll('input[type="password"]');
    
    passwordFields.forEach(field => {
        const wrapper = document.createElement('div');
        wrapper.className = 'password-field';
        
        field.parentNode.insertBefore(wrapper, field);
        wrapper.appendChild(field);
        
        const toggleButton = document.createElement('button');
        toggleButton.type = 'button';
        toggleButton.className = 'password-toggle';
        toggleButton.innerHTML = '<i class="fas fa-eye"></i>';
        toggleButton.title = 'Mostrar/Ocultar contraseña';
        
        toggleButton.addEventListener('click', function() {
            const type = field.getAttribute('type') === 'password' ? 'text' : 'password';
            field.setAttribute('type', type);
            
            const icon = this.querySelector('i');
            icon.classList.toggle('fa-eye');
            icon.classList.toggle('fa-eye-slash');
        });
        
        wrapper.appendChild(toggleButton);
    });
}

/**
 * Inicializa la validación general del formulario
 */
function initializeFormValidation() {
    const requiredFields = document.querySelectorAll('input[required], select[required]');
    
    requiredFields.forEach(field => {
        // Agregar asterisco a la etiqueta
        const label = document.querySelector(`label[for="${field.id}"]`);
        if (label && !label.classList.contains('required')) {
            label.classList.add('required');
        }
        
        // Validación en tiempo real
        field.addEventListener('blur', function() {
            validateField(this);
        });
        
        field.addEventListener('input', function() {
            if (this.classList.contains('is-invalid')) {
                validateField(this);
            }
        });
    });
    
    // Validación de email
    const emailField = document.getElementById('email');
    if (emailField) {
        emailField.addEventListener('blur', function() {
            validateEmail(this);
        });
    }
}

/**
 * Valida un campo individual
 * @param {HTMLElement} field - Campo a validar
 */
function validateField(field) {
    const value = field.value.trim();
    field.classList.remove('is-valid', 'is-invalid');
    removeFieldMessage(field);
    
    if (field.hasAttribute('required') && !value) {
        field.classList.add('is-invalid');
        setFieldMessage(field, 'Este campo es obligatorio', 'invalid');
        return false;
    }
    
    if (value) {
        field.classList.add('is-valid');
    }
    
    return true;
}

/**
 * Valida el formato del email
 * @param {HTMLElement} emailField - Campo de email
 */
function validateEmail(emailField) {
    const email = emailField.value.trim();
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    
    emailField.classList.remove('is-valid', 'is-invalid');
    removeFieldMessage(emailField);
    
    if (email && !emailRegex.test(email)) {
        emailField.classList.add('is-invalid');
        setFieldMessage(emailField, 'El formato del email no es válido', 'invalid');
        return false;
    }
    
    if (email) {
        emailField.classList.add('is-valid');
    }
    
    return true;
}

/**
 * Establece un mensaje de validación para un campo
 * @param {HTMLElement} field - Campo
 * @param {string} message - Mensaje
 * @param {string} type - Tipo (valid/invalid)
 */
function setFieldMessage(field, message, type) {
    removeFieldMessage(field);
    
    const messageDiv = document.createElement('div');
    messageDiv.className = type === 'valid' ? 'valid-feedback' : 'invalid-feedback';
    messageDiv.textContent = message;
    
    field.parentNode.appendChild(messageDiv);
}

/**
 * Remueve mensajes de validación de un campo
 * @param {HTMLElement} field - Campo
 */
function removeFieldMessage(field) {
    const existingMessage = field.parentNode.querySelector('.valid-feedback, .invalid-feedback');
    if (existingMessage) {
        existingMessage.remove();
    }
}

/**
 * Configura el envío del formulario
 */
function setupFormSubmission() {
    const form = document.querySelector('form');
    if (!form) return;
    
    form.addEventListener('submit', function(e) {
        if (!validateForm()) {
            e.preventDefault();
            
            if (typeof Swal !== 'undefined') {
                Swal.fire({
                    icon: 'error',
                    title: 'Formulario incompleto',
                    text: 'Por favor, complete todos los campos requeridos correctamente.'
                });
            } else {
                alert('Por favor, complete todos los campos requeridos correctamente.');
            }
        }
    });
}

/**
 * Valida todo el formulario
 * @returns {boolean} - True si es válido
 */
function validateForm() {
    let isValid = true;
    
    // Validar campos requeridos
    const requiredFields = document.querySelectorAll('input[required], select[required]');
    requiredFields.forEach(field => {
        if (!validateField(field)) {
            isValid = false;
        }
    });
    
    // Validar email
    const emailField = document.getElementById('email');
    if (emailField && !validateEmail(emailField)) {
        isValid = false;
    }
    
    // Validar coincidencia de contraseñas
    const passwordField = document.getElementById('password');
    const confirmPasswordField = document.getElementById('password_confirmation');
    
    if (passwordField && confirmPasswordField) {
        if (passwordField.value !== confirmPasswordField.value) {
            confirmPasswordField.classList.add('is-invalid');
            setFieldMessage(confirmPasswordField, 'Las contraseñas no coinciden', 'invalid');
            isValid = false;
        }
    }
    
    return isValid;
}
