<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use App\Models\TipoSolicitud;
use App\Models\Organismo;
use App\Models\Solicitud;
use App\Services\NotificationService;

class SolicitudController extends Controller
{
    // Mostrar formulario de creación de solicitud
    public function createDetails()
    {
        $tipos = TipoSolicitud::where('activo', true)->get();
        $organismos = Organismo::all();
        return view('solicitudes.create_details', compact('tipos', 'organismos'));
    }

    // Guardar la primera parte de la solicitud en sesión y pasar a la siguiente etapa
    public function storeDetails(Request $request)
    {
        // Verificar si el formulario fue enviado completamente
        if (empty($_POST) && empty($_FILES) && $_SERVER['CONTENT_LENGTH'] > 0) {
            $displayMaxSize = ini_get('post_max_size');
            return back()->with('error', "El formulario excede el tamaño máximo permitido ({$displayMaxSize}). Por favor, reduzca el número o tamaño de los archivos.");
        }

        try {
            $validated = $request->validate([
                'tipo_solicitud_id' => 'required|exists:tipo_solicitudes,id',
                'organismo_requeridor_id' => 'required|exists:organismos,id',
                'notificacion' => 'required|boolean',
                'detalle_solicitud' => 'required|string|max:4000',
                'observaciones' => 'nullable|string|max:1000',
                'evidencia.*' => 'nullable|file|extensions:jpg,jpeg,png,gif,pdf,doc,docx|max:10240',
            ], [
                'tipo_solicitud_id.required' => 'Debe seleccionar un tipo de solicitud.',
                'tipo_solicitud_id.exists' => 'El tipo de solicitud seleccionado no es válido.',
                'organismo_requeridor_id.required' => 'Debe seleccionar un organismo requerido.',
                'organismo_requeridor_id.exists' => 'El organismo seleccionado no es válido.',
                'notificacion.required' => 'Debe indicar si desea recibir notificaciones.',
                'detalle_solicitud.required' => 'El detalle de la solicitud es obligatorio.',
                'detalle_solicitud.max' => 'El detalle de la solicitud no puede exceder 12000 caracteres.',
                'observaciones.max' => 'Las observaciones no pueden exceder 1000 caracteres.',
                'evidencia.*.file' => 'Uno de los archivos seleccionados no es válido.',
                'evidencia.*.extensions' => 'Los archivos deben tener extensión: JPG, JPEG, PNG, GIF, PDF, DOC o DOCX.',
                'evidencia.*.max' => 'Cada archivo no puede exceder 10MB de tamaño.',
            ]);
            
            unset($validated['evidencia']);
            
        } catch (\Illuminate\Validation\ValidationException $e) {
            Log::error('storeDetails - Error de validación:', [
                'errors' => $e->errors(),
                'input' => $request->except(['evidencia'])
            ]);
            return back()->withInput()->with('error', 'Error en la validación: ' . implode(', ', $e->validator->errors()->all()));
        }
        
        // Validación adicional para cantidad de archivos
        if ($request->hasFile('evidencia') && count($request->file('evidencia')) > 10) {
            return back()->withInput()->with('error', 'No se pueden subir más de 10 archivos de evidencia.');
        }
        
        if ($request->hasFile('evidencia')) {
            foreach ($request->file('evidencia') as $index => $file) {
                
                if (!$file->isValid()) {
                    return back()->withInput()->with('error', "Error en el archivo {$file->getClientOriginalName()}: " . $file->getErrorMessage());
                }
                
                // Validar que el tipo MIME coincida con la extensión
                $mimeType = $file->getMimeType();
                $extension = strtolower($file->getClientOriginalExtension());
                
                // Detectar archivos WebP disfrazados
                if ($mimeType === 'image/webp' && $extension !== 'webp') {
                    return back()->withInput()->with('error', "El archivo {$file->getClientOriginalName()} es un archivo WebP con extensión incorrecta. WebP no está soportado. Por favor use JPG, PNG o GIF.");
                }
                
                // Validar que el MIME type sea coherente con la extensión
                $allowedMimes = [
                    'jpg' => ['image/jpeg'],
                    'jpeg' => ['image/jpeg'],
                    'png' => ['image/png'],
                    'gif' => ['image/gif'],
                    'pdf' => ['application/pdf'],
                    'doc' => ['application/msword'],
                    'docx' => ['application/vnd.openxmlformats-officedocument.wordprocessingml.document']
                ];
                
                if (isset($allowedMimes[$extension]) && !in_array($mimeType, $allowedMimes[$extension])) {
                    return back()->withInput()->with('error', "El archivo {$file->getClientOriginalName()} no parece ser un archivo {$extension} válido. Tipo detectado: {$mimeType}");
                }
            }
        }
        
        try {
            // Procesar archivos de evidencia si existen
            $evidenciaFiles = [];
            $tempSolicitudId = null;
            
            if ($request->hasFile('evidencia')) {
                // Generar un ID temporal para la solicitud
                $tempSolicitudId = 'temp_' . uniqid() . '_' . time();
                $basePath = env('EVIDENCIAS_STORAGE_PATH', storage_path('evidencias'));
                $tempSolicitudPath = $basePath . '/' . $tempSolicitudId;
                
                // Crear directorio temporal para esta solicitud
                if (!file_exists($tempSolicitudPath)) {
                    mkdir($tempSolicitudPath, 0755, true);
                }
                
                foreach ($request->file('evidencia') as $index => $file) {
                    // Obtener información del archivo ANTES de moverlo
                    $originalName = $file->getClientOriginalName();
                    $fileSize = $file->getSize();
                    $mimeType = $file->getMimeType();
                    
                    // Generar nombre único para el archivo
                    $extension = pathinfo($originalName, PATHINFO_EXTENSION);
                    $fileName = sprintf('%d_%s_%s.%s', 
                        $index + 1,
                        date('Y-m-d_H-i-s'),
                        preg_replace('/[^a-zA-Z0-9_-]/', '_', pathinfo($originalName, PATHINFO_FILENAME)),
                        $extension
                    );
                    
                    // Guardar archivo directamente en la carpeta temporal definitiva
                    $filePath = $tempSolicitudPath . '/' . $fileName;
                    $file->move($tempSolicitudPath, $fileName);
                    
                    // Establecer permisos del archivo
                    chmod($filePath, 0644);
                    
                    $evidenciaFiles[] = [
                        'original_name' => $originalName,
                        'size' => $fileSize,
                        'mime_type' => $mimeType,
                        'file_name' => $fileName,
                        'file_path' => $filePath,
                    ];
                }
            }
            
            // Guardar SOLO la información de archivos en sesión (NO los objetos UploadedFile)
            session(['solicitud.evidencia_files' => $evidenciaFiles]);
            
            // Guardar el ID temporal de la solicitud para poder renombrar la carpeta después
            if ($tempSolicitudId) {
                session(['solicitud.temp_solicitud_id' => $tempSolicitudId]);
            }
            
            // Guardar datos validados en sesión
            session(['solicitud.details' => $validated]);
            
            // Redirigir a la siguiente etapa del formulario
            return redirect()->route('solicitudes.create.contact');
            
        } catch (\Exception $e) {
            Log::error('storeDetails - Error al procesar archivos:', [
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            // Limpiar carpeta temporal si se creó y hubo error
            if (isset($tempSolicitudId)) {
                $this->cleanupTempFolder($tempSolicitudId);
            }
            
            return back()->withInput()->with('error', 'Error al procesar los archivos de evidencia: ' . $e->getMessage());
        }
    }


    public function createContact()
    {
        return view('solicitudes.create_contact', [
            'details' => session('solicitud.details', []),
        ]);
    }

    public function storeContact(Request $request)
    {
        try {
            $validated = $request->validate([
                'nombres' => 'required|string|min:2|max:255',
                'apellido_paterno' => 'required|string|min:2|max:255',
                'apellido_materno' => 'nullable|string|min:2|max:255',
                'nacionalidad' => 'nullable|string|min:2|max:100',
                'pais_residencia' => 'nullable|string|min:2|max:100',
                'calle' => 'nullable|string|min:2|max:255',
                'numero' => 'nullable|string|min:1|max:20',
                'villa' => 'nullable|string|min:2|max:100',
                'comuna' => 'nullable|string|min:2|max:100',
                'telefono' => 'nullable|string|min:6|max:30',
                'email' => 'nullable|email|min:5|max:255',
                'email_confirmation' => 'nullable|email|same:email|min:5',
            ], [
                'nombres.min' => 'Debe ingresar al menos 2 caracteres en nombres.',
                'apellido_paterno.min' => 'Debe ingresar al menos 2 caracteres en apellido paterno.',
                'apellido_materno.min' => 'Debe ingresar al menos 2 caracteres en apellido materno.',
                'nacionalidad.min' => 'Debe ingresar al menos 2 caracteres en nacionalidad.',
                'pais_residencia.min' => 'Debe ingresar al menos 2 caracteres en país de residencia.',
                'calle.min' => 'Debe ingresar al menos 2 caracteres en calle.',
                'numero.min' => 'Debe ingresar al menos 1 carácter en número.',
                'villa.min' => 'Debe ingresar al menos 2 caracteres en villa/población/departamento.',
                'comuna.min' => 'Debe ingresar al menos 2 caracteres en comuna.',
                'telefono.min' => 'Debe ingresar al menos 6 caracteres en teléfono.',
                'email.min' => 'Debe ingresar al menos 5 caracteres en email.',
                'email_confirmation.min' => 'Debe ingresar al menos 5 caracteres en la confirmación de email.',
                'email_confirmation.same' => 'Los emails deben coincidir.',
                'email.email' => 'Debe ser un email válido.',
                'email_confirmation.email' => 'La confirmación debe ser un email válido.',
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return back()->withInput()->withErrors($e->validator->errors());
        }
        
        unset($validated['email_confirmation']);
        
        session(['solicitud.contact' => $validated]);
        return redirect()->route('solicitudes.create.statistics');
    }


    public function createStatistics()
    {
        return view('solicitudes.create_statistics', [
            'details' => session('solicitud.details', []),
            'contact' => session('solicitud.contact', []),
        ]);
    }

    public function storeStatistics(Request $request)
    {
        try {
            $validated = $request->validate([
                'tipo_zona' => 'nullable|string|max:50',
                'educacion' => 'nullable|string|max:100',
                'edad' => 'nullable|integer|min:0|max:120',
                'ocupacion' => 'nullable|string|max:100',
                'genero' => 'nullable|string|max:50',
                'organizacion' => 'nullable|string|max:255',
                'apoderado' => 'nullable|string|max:255',
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return back()->withInput()->withErrors($e->validator->errors());
        }
        
        // Guardar en sesión
        session(['solicitud.statistics' => $validated]);
        // Redirigir a la etapa final de confirmación
        return redirect()->route('solicitudes.create.summary');
    }


    public function createSummary()
    {
        $details = session('solicitud.details', []);
        $contact = session('solicitud.contact', []);
        $statistics = session('solicitud.statistics', []);
        $evidenciaFiles = session('solicitud.evidencia_files', []);
        
        // Obtener nombres de tipo de solicitud y organismo
        $tipoSolicitud = null;
        $organismo = null;
        
        if (isset($details['tipo_solicitud_id'])) {
            $tipoSolicitud = TipoSolicitud::find($details['tipo_solicitud_id']);
        }
        
        if (isset($details['organismo_requeridor_id'])) {
            $organismo = Organismo::find($details['organismo_requeridor_id']);
        }
        
        return view('solicitudes.create_summary', [
            'details' => $details,
            'contact' => $contact,
            'statistics' => $statistics,
            'tipoSolicitud' => $tipoSolicitud,
            'organismo' => $organismo,
            'evidenciaFiles' => $evidenciaFiles,
        ]);
    }


    public function storeSummary(Request $request, NotificationService $notificationService)
    {
        try {
            $validated = $request->validate([
                'tipo_solicitud_id' => 'required|exists:tipo_solicitudes,id',
                'organismo_requeridor_id' => 'required|exists:organismos,id',
                'notificacion' => 'required|boolean',
                'detalle_solicitud' => 'required|string|max:12000',
                'fecha_solicitud' => 'nullable|date',
                'observaciones' => 'nullable|string|max:1000',
                'nombres' => 'required|string|max:255',
                'apellido_paterno' => 'required|string|max:255',
                'apellido_materno' => 'nullable|string|max:255',
                'nacionalidad' => 'nullable|string|max:100',
                'pais_residencia' => 'nullable|string|max:100',
                'calle' => 'nullable|string|max:255',
                'numero' => 'nullable|string|max:20',
                'villa' => 'nullable|string|max:100',
                'comuna' => 'nullable|string|max:100',
                'telefono' => 'nullable|string|max:30',
                'email' => 'nullable|email|max:255',
                'tipo_zona' => 'nullable|string|max:50',
                'educacion' => 'nullable|string|max:100',
                'edad' => 'nullable|integer|min:0|max:120',
                'ocupacion' => 'nullable|string|max:100',
                'genero' => 'nullable|string|max:50',
                'organizacion' => 'nullable|string|max:255',
                'apoderado' => 'nullable|string|max:255',
                'usuario_respuesta' => 'nullable|string|max:255',
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return back()->withInput()->withErrors($e->validator->errors());
        }
        
        $validated['fecha_solicitud'] = now();
        $validated['estado'] = 'pendiente';
        
        // Generar código de seguimiento único
        $validated['codigo_seguimiento'] = Solicitud::generarCodigoSeguimiento();
        
        try {
            // Crear la solicitud primero para obtener el ID
            $solicitud = Solicitud::create($validated);
                        
            // Procesar evidencias si existen
            $evidenciaFiles = session('solicitud.evidencia_files', []);
            $tempSolicitudId = session('solicitud.temp_solicitud_id');
            
            if (!empty($evidenciaFiles) && $tempSolicitudId) {
                $uriEvidencia = $this->renameEvidenciaFolder($solicitud->id, $tempSolicitudId);
                $solicitud->update(['uri_evidencia' => $uriEvidencia]);
                
            }
            
            // Limpiar la sesión
            session()->forget(['solicitud.details', 'solicitud.contact', 'solicitud.statistics', 'solicitud.evidencia_files', 'solicitud.temp_solicitud_id']);
            
            // Enviar notificación por correo si el usuario lo solicitó
            try {
                if ($solicitud->notificacion && $solicitud->email) {
                    $notificationService->notificarCreacionSolicitud($solicitud);
                    Log::info("Notificación de creación enviada para solicitud #{$solicitud->id}");
                }
            } catch (\Exception $e) {
                Log::warning("No se pudo enviar notificación de creación para solicitud #{$solicitud->id}: " . $e->getMessage());
            }
            
            return redirect()->route('solicitudes.enviada')
                ->with('success', 'Solicitud enviada correctamente.');
        } catch (\Exception $e) {
            Log::error('storeSummary - Error:', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);
            
            // Limpiar carpeta temporal si existe
            $tempSolicitudId = session('solicitud.temp_solicitud_id');
            if ($tempSolicitudId) {
                $this->cleanupTempFolder($tempSolicitudId);
            }
            
            return back()->withInput()->with('error', 'Ocurrió un error al guardar la solicitud. Intente nuevamente.');
        }
    }
    
    /**
     * Renombrar la carpeta temporal de evidencias con el ID real de la solicitud
     */
    private function renameEvidenciaFolder($solicitudId, $tempSolicitudId)
    {
        $basePath = env('EVIDENCIAS_STORAGE_PATH', storage_path('evidencias'));
        $tempPath = $basePath . '/' . $tempSolicitudId;
        $finalPath = $basePath . '/' . $solicitudId;
        
        if (file_exists($tempPath)) {
            $renamed = rename($tempPath, $finalPath);
            
            if ($renamed) {
                Log::info("renameEvidenciaFolder - Carpeta renombrada exitosamente de {$tempPath} a {$finalPath}");
                return $finalPath;
            } else {
                Log::error("renameEvidenciaFolder - Error al renombrar carpeta de {$tempPath} a {$finalPath}");
                throw new \Exception("No se pudo renombrar la carpeta de evidencias");
            }
        } else {
            Log::warning("renameEvidenciaFolder - Carpeta temporal no encontrada: {$tempPath}");
            return null;
        }
    }
    
    /**
     * Limpiar carpeta temporal en caso de error
     */
    private function cleanupTempFolder($tempSolicitudId)
    {
        $basePath = env('EVIDENCIAS_STORAGE_PATH', storage_path('evidencias'));
        $tempPath = $basePath . '/' . $tempSolicitudId;
        
        Log::info("cleanupTempFolder - Limpiando carpeta temporal:", [
            'temp_solicitud_id' => $tempSolicitudId,
            'temp_path' => $tempPath,
            'exists' => file_exists($tempPath)
        ]);
        
        if (file_exists($tempPath)) {
            // Eliminar todos los archivos dentro de la carpeta
            $files = scandir($tempPath);
            foreach ($files as $file) {
                if ($file !== '.' && $file !== '..' && is_file($tempPath . '/' . $file)) {
                    unlink($tempPath . '/' . $file);
                    Log::info("cleanupTempFolder - Archivo eliminado: {$file}");
                }
            }
            
            // Eliminar la carpeta
            $removed = rmdir($tempPath);
            
            if ($removed) {
                Log::info("cleanupTempFolder - Carpeta temporal eliminada exitosamente: {$tempPath}");
            } else {
                Log::error("cleanupTempFolder - Error al eliminar carpeta temporal: {$tempPath}");
            }
        }
    }
    
    /**
     * Buscar solicitud por código de seguimiento
     */
    public function buscarSeguimiento(Request $request)
    {
        $request->validate([
            'codigo_seguimiento' => 'required|string|min:12|max:20',
        ], [
            'codigo_seguimiento.required' => 'Debe ingresar un código de seguimiento.',
            'codigo_seguimiento.min' => 'El código de seguimiento debe tener al menos 12 caracteres.',
            'codigo_seguimiento.max' => 'El código de seguimiento no puede exceder 20 caracteres.',
        ]);

        $codigo = strtoupper(trim($request->codigo_seguimiento));
        $solicitud = Solicitud::where('codigo_seguimiento', $codigo)->first();

        if (!$solicitud) {
            return back()->with('error', 'No se encontró ninguna solicitud con el código de seguimiento: ' . $codigo . '. Verifique que el código esté escrito correctamente.');
        }

        return redirect()->route('seguimiento.mostrar', $codigo);
    }

    /**
     * Mostrar detalle de seguimiento de solicitud
     */
    public function mostrarSeguimiento($codigo)
    {
        $solicitud = Solicitud::with(['tipoSolicitud', 'organismoRequeridor', 'gestiones.usuario'])
            ->where('codigo_seguimiento', $codigo)
            ->first();

        if (!$solicitud) {
            abort(404, 'Solicitud no encontrada');
        }

        // Obtener archivos de cierre si existen
        $archivosCierre = [];
        if ($solicitud->uri_archivos_cierre && file_exists($solicitud->uri_archivos_cierre)) {
            $archivosCierre = $solicitud->getArchivosCierre();
        }

        return view('seguimiento.detalle', compact('solicitud', 'archivosCierre'));
    }

    /**
     * Descargar archivo de cierre desde seguimiento público
     */
    public function descargarArchivoCierre($codigo, $filename)
    {
        $solicitud = Solicitud::where('codigo_seguimiento', $codigo)->first();

        if (!$solicitud) {
            abort(404, 'Solicitud no encontrada');
        }

        if (!$solicitud->uri_archivos_cierre || !file_exists($solicitud->uri_archivos_cierre)) {
            abort(404, 'Archivos de cierre no encontrados');
        }

        $filePath = $solicitud->uri_archivos_cierre . '/' . $filename;

        // Validar que el archivo existe y está dentro de la carpeta correcta
        if (!file_exists($filePath) || !str_starts_with(realpath($filePath), realpath($solicitud->uri_archivos_cierre))) {
            abort(404, 'Archivo no encontrado');
        }

        // Determinar el tipo MIME del archivo
        $mimeType = mime_content_type($filePath);

        return response()->download($filePath, $filename, [
            'Content-Type' => $mimeType,
        ]);
    }
}
