<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Symfony\Component\HttpFoundation\Response;

class ValidateFormStep
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next, string $requiredStep): Response
    {
        // Definir el orden de los pasos y sus dependencias
        $stepHierarchy = [
            'details' => [],
            'contact' => ['details'],
            'statistics' => ['details', 'contact'],
            'summary' => ['details', 'contact', 'statistics']
        ];

        // Verificar que el paso requerido existe
        if (!array_key_exists($requiredStep, $stepHierarchy)) {
            return redirect()->route('solicitudes.create.details')
                ->with('error', 'Paso del formulario no válido.');
        }

        // Obtener los datos de la sesión
        $sessionData = session('solicitud', []);

        // Verificar que se han completado todos los pasos prerrequisitos
        foreach ($stepHierarchy[$requiredStep] as $prerequisiteStep) {
            if (!isset($sessionData[$prerequisiteStep]) || empty($sessionData[$prerequisiteStep])) {
                // Determinar a qué paso redirigir (el primer paso incompleto)
                $redirectStep = $this->getFirstIncompleteStep($sessionData, $stepHierarchy);
                
                $message = 'Debe completar los pasos anteriores antes de continuar.';
                
                // Mensajes específicos según el paso
                switch ($redirectStep) {
                    case 'details':
                        $message = 'Debe completar los detalles de la solicitud primero.';
                        break;
                    case 'contact':
                        $message = 'Debe completar sus datos de contacto antes de continuar.';
                        break;
                    case 'statistics':
                        $message = 'Debe completar los datos estadísticos antes de continuar.';
                        break;
                }
                
                return redirect()->route("solicitudes.create.{$redirectStep}")
                    ->with('warning', $message);
            }
        }

        // Validación adicional para evitar manipulación de sesión
        if ($requiredStep !== 'details') {
            $this->validateSessionIntegrity($sessionData, $requiredStep);
        }

        return $next($request);
    }

    /**
     * Encontrar el primer paso incompleto
     */
    private function getFirstIncompleteStep(array $sessionData, array $stepHierarchy): string
    {
        foreach ($stepHierarchy as $step => $prerequisites) {
            if (!isset($sessionData[$step]) || empty($sessionData[$step])) {
                return $step;
            }
        }
        
        return 'details'; // Fallback
    }

    /**
     * Validar integridad de la sesión
     */
    private function validateSessionIntegrity(array $sessionData, string $currentStep): void
    {
        // Verificar que los datos de sesión tienen las claves esperadas
        $requiredKeys = [
            'details' => ['tipo_solicitud_id', 'organismo_requeridor_id', 'detalle_solicitud'],
            'contact' => ['nombres', 'apellido_paterno'],
            'statistics' => [] // Los datos estadísticos son opcionales
        ];

        foreach ($requiredKeys as $step => $keys) {
            if (isset($sessionData[$step]) && !empty($sessionData[$step])) {
                foreach ($keys as $key) {
                    if (!isset($sessionData[$step][$key]) || empty($sessionData[$step][$key])) {
                        // Si faltan datos críticos, limpiar la sesión y redirigir
                        session()->forget('solicitud');
                        redirect()->route('solicitudes.create.details')
                            ->with('error', 'Se detectaron datos incompletos. Por favor, complete el formulario nuevamente.')
                            ->send();
                        exit;
                    }
                }
            }
        }
    }
}
