// JavaScript para la vista de índice de solicitudes

// Función para mostrar vista previa de imagen
function previewImage(imageUrl, imageName) {
    const modalHtml = `
        <div class="modal fade image-preview-modal" id="imagePreviewModal" tabindex="-1" role="dialog" aria-labelledby="imagePreviewLabel" aria-hidden="true">
            <div class="modal-dialog modal-lg" role="document">
                <div class="modal-content">
                    <div class="modal-header bg-dark text-white">
                        <h5 class="modal-title" id="imagePreviewLabel">
                            <i class="fas fa-image mr-2"></i>Vista previa: ${imageName}
                        </h5>
                        <button type="button" class="close text-white" data-dismiss="modal" aria-label="Cerrar">
                            <span aria-hidden="true">&times;</span>
                        </button>
                    </div>
                    <div class="modal-body text-center p-0">
                        <div class="loading-evidencias d-none" id="imageLoading">
                            <div class="spinner-border text-primary" role="status">
                                <span class="sr-only">Cargando...</span>
                            </div>
                            <div>Cargando imagen...</div>
                        </div>
                        <img id="previewImage" src="" class="img-fluid d-none" alt="${imageName}" style="max-height: 70vh; border-radius: 8px;">
                    </div>
                    <div class="modal-footer bg-light">
                        <a href="${imageUrl}" class="btn btn-primary" download="${imageName}">
                            <i class="fas fa-download mr-1"></i>Descargar
                        </a>
                        <button type="button" class="btn btn-secondary" data-dismiss="modal">
                            <i class="fas fa-times mr-1"></i>Cerrar
                        </button>
                    </div>
                </div>
            </div>
        </div>
    `;
    
    $('#imagePreviewModal').remove();
    $('body').append(modalHtml);
    $('#imagePreviewModal').css('z-index', 1080);
    $('#imagePreviewModal').modal('show');
    
    $('#imageLoading').removeClass('d-none');
    const img = document.getElementById('previewImage');
    
    img.onload = function() {
        $('#imageLoading').addClass('d-none');
        $('#previewImage').removeClass('d-none');
    };
    
    img.onerror = function() {
        $('#imageLoading').addClass('d-none');
        $('#previewImage').removeClass('d-none').attr('src', 'data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjAwIiBoZWlnaHQ9IjE1MCIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KICA8cmVjdCB3aWR0aD0iMTAwJSIgaGVpZ2h0PSIxMDAlIiBmaWxsPSIjZGRkIi8+CiAgPHRleHQgeD0iNTAlIiB5PSI1MCUiIGZvcnQtZmFtaWx5PSJBcmlhbCIgZm9udC1zaXplPSIxNCIgZmlsbD0iIzk5OSIgdGV4dC1hbmNob3I9Im1pZGRsZSIgZHk9Ii4zZW0iPkVycm9yIGFsIGNhcmdhciBpbWFnZW48L3RleHQ+Cjwvc3ZnPg==').attr('alt', 'Error al cargar imagen');
    };
    
    img.src = imageUrl;
    
    $('#imagePreviewModal').on('hidden.bs.modal', function () {
        $(this).remove();
    });
}

// Función para mostrar información de archivo
function showFileInfo(fileName, fileSize, fileType) {
    const sizeInMB = (fileSize / (1024 * 1024)).toFixed(2);
    const sizeInKB = (fileSize / 1024).toFixed(1);
    const displaySize = fileSize > 1024 * 1024 ? `${sizeInMB} MB` : `${sizeInKB} KB`;
    
    return `
        <div class="file-info">
            <strong>Archivo:</strong> ${fileName}<br>
            <strong>Tamaño:</strong> ${displaySize}<br>
            <strong>Tipo:</strong> ${fileType}
        </div>
    `;
}

// Función para copiar enlace de descarga
function copyDownloadLink(url) {
    navigator.clipboard.writeText(url).then(function() {
        if (typeof Swal !== 'undefined') {
            Swal.fire({
                icon: 'success',
                title: '¡Enlace copiado!',
                text: 'El enlace de descarga se ha copiado al portapapeles',
                timer: 2000,
                showConfirmButton: false,
                toast: true,
                position: 'top-end'
            });
        }
    }).catch(function(err) {
        console.error('Error al copiar enlace: ', err);
    });
}

// Función para descargar todas las evidencias
function downloadAllEvidences(solicitudId) {
    if (confirm('¿Desea descargar todas las evidencias de esta solicitud?')) {
        window.location.href = `/admin/solicitudes/${solicitudId}/evidencias/download-all`;
    }
}

// Inicialización cuando el DOM esté listo
$(document).ready(function() {
    // Configuración de modales
    initModals();
    
    // Configuración de tooltips
    initTooltips();
    
    // Configuración del cierre de solicitudes
    initCerrarSolicitudes();
    
    // Configuración de teclas rápidas
    initKeyboardShortcuts();
    
    // Configuración de lazy loading para evidencias
    initEvidenciasLazyLoading();
    
    fixModalZIndex(); // Agregar esta línea
});

// Función para inicializar modales
function initModals() {
    $('.modal').on('show.bs.modal', function() {
        const $modal = $(this);
        const $backdrop = $('.modal-backdrop');
        
        $modal.css({
            'z-index': '9999',
            'position': 'fixed',
            'top': '0',
            'left': '0',
            'width': '100%',
            'height': '100%',
            'display': 'flex !important',
            'align-items': 'center',
            'justify-content': 'center'
        });
        
        $backdrop.css({
            'z-index': '9998',
            'position': 'fixed',
            'top': '0',
            'left': '0',
            'width': '100%',
            'height': '100%'
        });
        
        $modal.find('.modal-dialog').css({
            'z-index': '10000',
            'position': 'relative',
            'margin': 'auto'
        });
        
        $modal.find('.modal-content').css({
            'z-index': '10001',
            'position': 'relative'
        });
        
        $modal.appendTo('body');
    });
    
    $('.modal').on('shown.bs.modal', function() {
        const $modal = $(this);
        $modal.css('display', 'flex');
    });
    
    $('.modal').on('hidden.bs.modal', function() {
        if ($('.modal:visible').length) {
            $('body').addClass('modal-open');
        }
        $(this).removeAttr('style');
    });
    
    // Agregar loading overlay para modales
    $('.btn[data-toggle="modal"]').click(function() {
        const modalId = $(this).attr('data-target');
        $(modalId).find('.modal-body').prepend(`
            <div class="loading-overlay d-flex justify-content-center align-items-center" 
                 style="position: absolute; top: 0; left: 0; right: 0; bottom: 0; background: rgba(255,255,255,0.8); z-index: 1000;">
                <div class="spinner-border text-primary" role="status">
                    <span class="sr-only">Cargando...</span>
                </div>
            </div>
        `);
        
        setTimeout(() => {
            $(modalId).find('.loading-overlay').fadeOut(300, function() {
                $(this).remove();
            });
        }, 500);
    });
}

// Función para inicializar tooltips
function initTooltips() {
    $('[title]').tooltip({
        placement: 'auto',
        trigger: 'hover'
    });
}

// Función para inicializar el cierre de solicitudes
function initCerrarSolicitudes() {
    $('.cerrar-solicitud-btn').click(function(e) {
        e.preventDefault();
        e.stopPropagation();
        
        // Asegurar que cualquier modal abierto se cierre primero
        $('.modal').modal('hide');
        
        const solicitudId = $(this).data('solicitud-id');
        const solicitudNumero = $(this).data('solicitud-numero');
        
        Swal.fire({
            title: `¿Cerrar solicitud ${solicitudNumero}?`,
            html: `
                <div class="text-left">
                    <p class="mb-3">Esta acción marcará la solicitud como finalizada.</p>
                    <label for="observaciones_cierre" class="form-label text-muted font-weight-semibold">
                        <i class="fas fa-comment-alt mr-1"></i>Observaciones de cierre <span class="text-danger">*</span>
                    </label>
                    <textarea id="observaciones_cierre" class="form-control" rows="4" 
                              placeholder="Describa las acciones realizadas y el resultado final..." 
                              maxlength="1000" required></textarea>
                    <small class="text-muted">Máximo 1000 caracteres</small>
                </div>
            `,
            icon: 'warning',
            showCancelButton: true,
            confirmButtonColor: '#dc3545',
            cancelButtonColor: '#6c757d',
            confirmButtonText: '<i class="fas fa-check mr-1"></i>Sí, cerrar solicitud',
            cancelButtonText: '<i class="fas fa-times mr-1"></i>Cancelar',
            customClass: {
                container: 'cerrar-solicitud-modal'
            },
            zIndex: 10010, // Asegurar que SweetAlert esté por encima
            preConfirm: () => {
                const observaciones = document.getElementById('observaciones_cierre').value.trim();
                if (!observaciones) {
                    Swal.showValidationMessage('Las observaciones de cierre son obligatorias');
                    return false;
                }
                if (observaciones.length < 10) {
                    Swal.showValidationMessage('Las observaciones deben tener al menos 10 caracteres');
                    return false;
                }
                return observaciones;
            },
            didOpen: () => {
                document.getElementById('observaciones_cierre').focus();
                
                const textarea = document.getElementById('observaciones_cierre');
                const small = textarea.nextElementSibling;
                
                textarea.addEventListener('input', function() {
                    const remaining = 1000 - this.value.length;
                    small.textContent = `${remaining} caracteres restantes`;
                    small.className = remaining < 100 ? 'text-warning' : 'text-muted';
                });
            }
        }).then((result) => {
            if (result.isConfirmed) {
                Swal.fire({
                    title: 'Cerrando solicitud...',
                    html: 'Por favor espere mientras se procesa la solicitud',
                    allowOutsideClick: false,
                    allowEscapeKey: false,
                    didOpen: () => {
                        Swal.showLoading();
                    }
                });
                
                const form = $('<form>', {
                    method: 'POST',
                    action: `/admin/solicitudes/${solicitudId}/cerrar`
                });
                
                form.append($('<input>', {
                    type: 'hidden',
                    name: '_token',
                    value: $('meta[name="csrf-token"]').attr('content')
                }));
                
                form.append($('<input>', {
                    type: 'hidden',
                    name: 'observaciones_cierre',
                    value: result.value
                }));
                
                $('body').append(form);
                form.submit();
            }
        });
    });
}

// Función para inicializar teclas rápidas
function initKeyboardShortcuts() {
    $(document).on('keydown', function(e) {
        if ($('#imagePreviewModal').hasClass('show')) {
            if (e.key === 'Escape') {
                $('#imagePreviewModal').modal('hide');
            }
            if (e.ctrlKey && e.key === 's') {
                e.preventDefault();
                const downloadLink = $('#imagePreviewModal').find('a[download]');
                if (downloadLink.length) {
                    downloadLink[0].click();
                }
            }
        }
    });
}

// Función para inicializar lazy loading de evidencias
function initEvidenciasLazyLoading() {
    $('.evidencia-item').each(function() {
        const $item = $(this);
        const downloadUrl = $item.find('a[href]').attr('href');
        
        if (downloadUrl && (downloadUrl.includes('.jpg') || downloadUrl.includes('.png') || downloadUrl.includes('.jpeg'))) {
            $item.attr('data-image-url', downloadUrl);
        }
    });
    
    $('.evidencia-item').hover(
        function() {
            $(this).addClass('shadow-sm');
        },
        function() {
            $(this).removeClass('shadow-sm');
        }
    );
}

// Función para corregir problemas de z-index en modales
function fixModalZIndex() {
    // Corregir z-index cuando se abre un modal
    $(document).on('show.bs.modal', '.modal', function() {
        const zIndex = 9999 + $('.modal:visible').length * 10;
        $(this).css('z-index', zIndex);
        
        // Asegurar que el backdrop esté por debajo
        setTimeout(() => {
            $('.modal-backdrop').css('z-index', zIndex - 1);
        }, 50);
    });
    
    // Restaurar z-index cuando se cierra un modal
    $(document).on('hidden.bs.modal', '.modal', function() {
        $('.modal').each(function(index) {
            $(this).css('z-index', 9999 + index * 10);
        });
    });
    
    // Corregir z-index específicamente para modales de solicitudes
    $(document).on('shown.bs.modal', '[id^="modalResumen"]', function() {
        $(this).css({
            'z-index': '9999',
            'position': 'fixed'
        });
        
        $(this).find('.modal-dialog').css({
            'z-index': '10000',
            'position': 'relative'
        });
        
        $(this).find('.modal-content').css({
            'z-index': '10001',
            'position': 'relative'
        });
        
        // Asegurar que todos los botones sean clickeables
        $(this).find('.btn, .form-control, .close').css({
            'z-index': '10003',
            'position': 'relative',
            'pointer-events': 'auto'
        });
    });
}
