/**
 * Toast Handler para SIAC
 * Maneja las notificaciones toast usando SweetAlert2
 */

// Configuración por defecto para toasts
const toastConfig = {
    position: 'top-end',
    showConfirmButton: false,
    timer: 4000,
    timerProgressBar: true,
    toast: true,
    didOpen: (toast) => {
        toast.addEventListener('mouseenter', Swal.stopTimer);
        toast.addEventListener('mouseleave', Swal.resumeTimer);
    }
};

/**
 * Mostrar toast personalizado
 * @param {Object} options - Opciones del toast
 * @param {string} options.text - Texto del mensaje
 * @param {string} options.icon - Tipo de icono (success, error, warning, info)
 * @param {number} options.timer - Duración en milisegundos
 * @param {string} options.position - Posición del toast
 */
function showToast(options = {}) {
    const config = {
        ...toastConfig,
        ...options
    };

    // Aplicar estilos específicos según el tipo
    switch (options.icon) {
        case 'success':
            config.background = '#d4edda';
            config.color = '#155724';
            config.iconColor = '#28a745';
            break;
        case 'error':
            config.background = '#f8d7da';
            config.color = '#721c24';
            config.iconColor = '#dc3545';
            break;
        case 'warning':
            config.background = '#fff3cd';
            config.color = '#856404';
            config.iconColor = '#ffc107';
            break;
        case 'info':
            config.background = '#d1ecf1';
            config.color = '#0c5460';
            config.iconColor = '#17a2b8';
            break;
    }

    Swal.fire(config);
}

/**
 * Toast de éxito
 * @param {string} text - Mensaje de éxito
 */
function showSuccessToast(text) {
    showToast({
        text: text,
        icon: 'success',
        timer: 4000
    });
}

/**
 * Toast de error
 * @param {string} text - Mensaje de error
 */
function showErrorToast(text) {
    showToast({
        text: text,
        icon: 'error',
        timer: 5000
    });
}

/**
 * Toast de advertencia
 * @param {string} text - Mensaje de advertencia
 */
function showWarningToast(text) {
    showToast({
        text: text,
        icon: 'warning',
        timer: 4500
    });
}

/**
 * Toast de información
 * @param {string} text - Mensaje informativo
 */
function showInfoToast(text) {
    showToast({
        text: text,
        icon: 'info',
        timer: 4000
    });
}

// Hacer las funciones globales
window.showToast = showToast;
window.showSuccessToast = showSuccessToast;
window.showErrorToast = showErrorToast;
window.showWarningToast = showWarningToast;
window.showInfoToast = showInfoToast;
