/**
 * =========================================
 * ADMIN ORGANISMOS - JavaScript Modular
 * ========================================= */

// Variables globales
let currentModal = null;

/**
 * Inicialización principal
 */
document.addEventListener('DOMContentLoaded', function() {
    initializeOrganismos();
    initializeSidebarToggle();
    showGlobalMessages();
});

/**
 * Inicializar toggle del sidebar
 */
function initializeSidebarToggle() {
    const sidebarToggle = document.querySelector('.sidebar-toggle');
    const sidebar = document.querySelector('.admin-sidebar');
    const mainContent = document.querySelector('.admin-main-content');
    
    if (sidebarToggle && sidebar) {
        sidebarToggle.addEventListener('click', function(e) {
            e.preventDefault();
            sidebar.classList.toggle('collapsed');
            
            // Ajustar margen del contenido principal
            if (sidebar.classList.contains('collapsed')) {
                mainContent.style.marginLeft = '0';
            } else {
                mainContent.style.marginLeft = '';
            }
        });
    }
}

/**
 * JavaScript para la gestión de organismos en el admin
 * Funcionalidad de jerarquía expandible/contraíble
 */

document.addEventListener('DOMContentLoaded', function() {
    initializeOrganismosHierarchy();
    initializeDeleteConfirmation();
    showGlobalMessages();
});

/**
 * Inicializa la funcionalidad de jerarquía de organismos
 */
function initializeOrganismosHierarchy() {
    console.log('Inicializando jerarquía de organismos...');
    
    const toggleButtons = document.querySelectorAll('.toggle-children');
    console.log(`Encontrados ${toggleButtons.length} botones de toggle`);
    
    toggleButtons.forEach(function(button, index) {
        console.log(`Configurando botón ${index + 1}:`, button);
        
        button.addEventListener('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            
            const organismoId = this.getAttribute('data-organismo-id');
            const level = parseInt(this.getAttribute('data-level'));
            const icon = this.querySelector('i');
            
            console.log(`Click en botón - Organismo ID: ${organismoId}, Nivel: ${level}`);
            
            toggleOrganismoChildren(organismoId, level, icon);
        });
    });
}

/**
 * Alterna la visibilidad de los hijos de un organismo
 * @param {string} organismoId - ID del organismo padre
 * @param {number} level - Nivel de jerarquía
 * @param {Element} icon - Icono del botón
 */
function toggleOrganismoChildren(organismoId, level, icon) {
    console.log(`Toggling children for organismo ${organismoId} at level ${level}`);
    
    // Encontrar todas las filas hijas directas de este organismo
    const allRows = document.querySelectorAll('tr[data-organismo-id]');
    const directChildren = [];
    
    // Buscar hijos directos
    allRows.forEach(function(row) {
        const rowLevel = parseInt(row.getAttribute('data-level'));
        const parentIds = row.getAttribute('data-parent-ids');
        
        // Si es nivel hijo directo
        if (rowLevel === level + 1) {
            if (level === 0) {
                // Para nivel 0, los hijos tienen parent_ids vacío y nivel 1
                if (!parentIds || parentIds === '') {
                    const rowOrganismoId = row.getAttribute('data-organismo-id');
                    // Verificar que realmente es hijo de este organismo
                    if (isDirectChild(organismoId, rowOrganismoId)) {
                        directChildren.push(row);
                    }
                }
            } else {
                // Para otros niveles, verificar la cadena de padres
                const parentIdsArray = parentIds ? parentIds.split(',').filter(id => id !== '') : [];
                if (parentIdsArray.length === level && parentIdsArray[parentIdsArray.length - 1] === organismoId) {
                    directChildren.push(row);
                }
            }
        }
    });
    
    console.log(`Encontrados ${directChildren.length} hijos directos`);
    
    // Verificar si los hijos están visibles
    let childrenVisible = false;
    if (directChildren.length > 0) {
        childrenVisible = directChildren[0].style.display !== 'none';
    }
    
    console.log(`Hijos actualmente visibles: ${childrenVisible}`);
    
    // Alternar visibilidad
    if (childrenVisible) {
        // Ocultar todos los descendientes
        hideAllDescendants(organismoId, level);
        // Cambiar icono a cerrado
        if (icon) {
            icon.classList.remove('fa-chevron-down');
            icon.classList.add('fa-chevron-right');
        }
    } else {
        // Mostrar solo los hijos directos
        directChildren.forEach(function(row) {
            row.style.display = 'table-row';
        });
        // Cambiar icono a abierto
        if (icon) {
            icon.classList.remove('fa-chevron-right');
            icon.classList.add('fa-chevron-down');
        }
    }
}

/**
 * Verifica si un organismo es hijo directo de otro
 */
function isDirectChild(parentId, childId) {
    // Esta función debería verificar en los datos reales
    // Por ahora usaremos una aproximación basada en la estructura HTML
    const childRow = document.querySelector(`tr[data-organismo-id="${childId}"]`);
    if (!childRow) return false;
    
    const level = parseInt(childRow.getAttribute('data-level'));
    const parentIds = childRow.getAttribute('data-parent-ids');
    
    if (level === 1 && (!parentIds || parentIds === '')) {
        // Es hijo directo de root
        return true;
    }
    
    if (parentIds) {
        const parentIdsArray = parentIds.split(',').filter(id => id !== '');
        return parentIdsArray[parentIdsArray.length - 1] === parentId;
    }
    
    return false;
}

/**
 * Oculta todos los descendientes de un organismo
 */
function hideAllDescendants(organismoId, parentLevel) {
    const allRows = document.querySelectorAll('tr[data-organismo-id]');
    
    allRows.forEach(function(row) {
        const rowLevel = parseInt(row.getAttribute('data-level'));
        const parentIds = row.getAttribute('data-parent-ids');
        
        if (rowLevel > parentLevel) {
            let isDescendant = false;
            
            if (parentLevel === 0) {
                // Para nivel 0, cualquier fila con nivel > 0 que tenga este ID en sus padres
                if (parentIds && parentIds.includes(organismoId)) {
                    isDescendant = true;
                }
            } else {
                // Para otros niveles, verificar la cadena completa
                if (parentIds) {
                    const parentIdsArray = parentIds.split(',').filter(id => id !== '');
                    isDescendant = parentIdsArray.includes(organismoId);
                }
            }
            
            if (isDescendant) {
                row.style.display = 'none';
                // También resetear iconos de botones anidados
                const nestedButton = row.querySelector('.toggle-children i');
                if (nestedButton) {
                    nestedButton.classList.remove('fa-chevron-down');
                    nestedButton.classList.add('fa-chevron-right');
                }
            }
        }
    });
}

/**
 * Inicializa la confirmación de eliminación
 */
function initializeDeleteConfirmation() {
    const deleteForms = document.querySelectorAll('.delete-organismo-form');
    
    deleteForms.forEach(function(form) {
        form.addEventListener('submit', function(e) {
            e.preventDefault();
            
            if (confirm('¿Estás seguro de que deseas eliminar este organismo? Esta acción no se puede deshacer.')) {
                this.submit();
            }
        });
    });
}

/**
 * Muestra mensajes globales si existen
 */
function showGlobalMessages() {
    // Mostrar mensaje de éxito si existe
    if (window.successMessage) {
        showNotification('Éxito', window.successMessage, 'success');
    }
    
    // Mostrar mensaje de error si existe
    if (window.errorMessage) {
        showNotification('Error', window.errorMessage, 'error');
    }
    
    // Mostrar errores de validación si existen
    if (window.errorMessages) {
        showNotification('Errores de validación', window.errorMessages, 'error');
    }
}

/**
 * Muestra una notificación
 */
function showNotification(title, message, type = 'info') {
    // Si existe SweetAlert, usarlo
    if (typeof Swal !== 'undefined') {
        Swal.fire({
            title: title,
            html: message,
            icon: type === 'error' ? 'error' : (type === 'success' ? 'success' : 'info'),
            timer: type === 'success' ? 3000 : 5000,
            showConfirmButton: type === 'error',
            toast: true,
            position: 'top-end'
        });
    } else {
        // Fallback a alert nativo
        alert(`${title}: ${message}`);
    }
}

/**
 * Carga el modal de visualización de organismo
 * @param {number} organismoId - ID del organismo
 */
function loadViewModal(organismoId) {
    console.log(`Cargando vista del organismo ${organismoId}`);
    
    if (typeof window.loadDynamicModal === 'function') {
        window.loadDynamicModal(`/admin/organismos/${organismoId}`, 'GET', 'Ver Organismo');
    } else {
        console.error('Función loadDynamicModal no disponible');
        // Fallback: navegar directamente
        window.location.href = `/admin/organismos/${organismoId}`;
    }
}

/**
 * Carga el modal de edición de organismo
 * @param {number} organismoId - ID del organismo
 */
function loadEditModal(organismoId) {
    console.log(`Cargando edición del organismo ${organismoId}`);
    
    if (typeof window.loadDynamicModal === 'function') {
        window.loadDynamicModal(`/admin/organismos/${organismoId}/edit`, 'GET', 'Editar Organismo');
    } else {
        console.error('Función loadDynamicModal no disponible');
        // Fallback: navegar directamente
        window.location.href = `/admin/organismos/${organismoId}/edit`;
    }
}
